import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, then pick up all the tomatoes and leave them on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list
        # Drawers: bottom, middle, top
        # Items: item1, item2, item3 (assume these are the tomatoes)
        # Plate: plate

        # For this plan, select the "bottom" drawer (could be any, but bottom is a safe default)
        # Get anchor/side/joint positions for the bottom drawer
        bottom_anchor_pos = positions.get('bottom_anchor_pos')
        bottom_side_pos = positions.get('bottom_side_pos')
        bottom_joint_pos = positions.get('bottom_joint_pos')
        # For pulling, we may use the joint position as the pull target

        # Get the plate position
        plate_pos = positions.get('plate')

        # Get tomato/item positions
        tomato_names = ['item1', 'item2', 'item3']
        tomato_positions = []
        for name in tomato_names:
            pos = positions.get(name)
            if pos is not None:
                tomato_positions.append((name, pos))
            else:
                print(f"[Warning] Could not find position for {name}.")

        # Step 1: Move to side of bottom drawer
        if bottom_side_pos is not None:
            print("[Task] Moving to side of bottom drawer.")
            obs, reward, done = move(env, task, target_pos=np.array(bottom_side_pos))
            if done:
                print("[Task] Task ended after move to side of drawer!")
                return
        else:
            print("[Error] bottom_side_pos not found.")
            return

        # Step 2: Rotate gripper to 90 degrees (for drawer grasp)
        # Quaternion for 90 degrees about z axis: [0, 0, sin(π/4), cos(π/4)]
        ninety_deg_quat = np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)])  # [x, y, z, w]
        print("[Task] Rotating gripper to 90 degrees.")
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 3: Move to anchor of bottom drawer
        if bottom_anchor_pos is not None:
            print("[Task] Moving to anchor of bottom drawer.")
            obs, reward, done = move(env, task, target_pos=np.array(bottom_anchor_pos))
            if done:
                print("[Task] Task ended after move to anchor!")
                return
        else:
            print("[Error] bottom_anchor_pos not found.")
            return

        # Step 4: Pick the drawer handle (simulate grasping the drawer)
        print("[Task] Picking (grasping) the drawer handle.")
        obs, reward, done = pick(env, task, target_pos=np.array(bottom_anchor_pos), approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended after picking drawer handle!")
            return

        # Step 5: Pull the drawer open (along x axis, positive direction)
        pull_distance = 0.20
        print("[Task] Pulling the drawer open.")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended after pulling drawer!")
            return

        # Step 6+: For each tomato, pick and place on plate
        for idx, (tomato_name, tomato_pos) in enumerate(tomato_positions):
            print(f"[Task] Processing {tomato_name} at {tomato_pos}.")

            # Move above the tomato (with a small offset in z)
            approach_pos = np.array(tomato_pos) + np.array([0, 0, 0.10])
            print(f"[Task] Moving above {tomato_name}.")
            obs, reward, done = move(env, task, target_pos=approach_pos)
            if done:
                print(f"[Task] Task ended after moving above {tomato_name}!")
                return

            # Rotate gripper to 0 deg (for vertical pick)
            zero_deg_quat = np.array([0, 0, 0, 1])  # Identity quaternion
            print(f"[Task] Rotating gripper to 0 degrees for picking {tomato_name}.")
            obs, reward, done = rotate(env, task, target_quat=zero_deg_quat)
            if done:
                print(f"[Task] Task ended after rotate for {tomato_name}!")
                return

            # Move to pick position (directly above tomato)
            print(f"[Task] Moving to pick position for {tomato_name}.")
            obs, reward, done = move(env, task, target_pos=np.array(tomato_pos))
            if done:
                print(f"[Task] Task ended after moving to pick position for {tomato_name}!")
                return

            # Pick the tomato
            print(f"[Task] Picking {tomato_name}.")
            obs, reward, done = pick(env, task, target_pos=np.array(tomato_pos), approach_distance=0.10, approach_axis='z')
            if done:
                print(f"[Task] Task ended after picking {tomato_name}!")
                return

            # Move to above plate
            plate_approach_pos = np.array(plate_pos) + np.array([0, 0, 0.10])
            print(f"[Task] Moving above plate to place {tomato_name}.")
            obs, reward, done = move(env, task, target_pos=plate_approach_pos)
            if done:
                print(f"[Task] Task ended after moving above plate for {tomato_name}!")
                return

            # Place the tomato on the plate
            print(f"[Task] Placing {tomato_name} on plate.")
            obs, reward, done = place(env, task, target_pos=np.array(plate_pos), approach_distance=0.10, approach_axis='z')
            if done:
                print(f"[Task] Task ended after placing {tomato_name}!")
                return

        print("[Task] All tomatoes placed on plate. Task completed successfully!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()