import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        try:
            bottom_side_pos = positions['bottom_side_pos']
            bottom_anchor_pos = positions['bottom_anchor_pos']
            item1_pos = positions['item1']
            item2_pos = positions['item2']
            plate_pos = positions['plate']
        except KeyError as e:
            print(f"[Error] Object position not found: {e}")
            shutdown_environment(env)
            return

        # === Drawer Opening Sequence ===
        # [Frozen Code Start]
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]))  # 90 deg about z
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        # [Frozen Code End]

        # Step 4: Pick the drawer handle (at anchor)
        print("[Step 4] Pick drawer handle (at anchor)")
        try:
            obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.10, approach_axis='-y')
        except Exception as e:
            print(f"[Error] Exception during pick at anchor: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after picking drawer handle!")
            shutdown_environment(env)
            return

        # Step 5: Pull to open the drawer
        print("[Step 5] Pull to open the drawer")
        try:
            obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after pull!")
            shutdown_environment(env)
            return

        # Step 6: Move to item1 (first tomato)
        print("[Step 6] Move to item1 (first tomato)")
        try:
            obs, reward, done = move(env, task, target_pos=item1_pos)
        except Exception as e:
            print(f"[Error] Exception during move to item1: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after move to item1!")
            shutdown_environment(env)
            return

        # Step 7: Rotate gripper to 0 deg (for object pick)
        print("[Step 7] Rotate gripper to 0 deg (object pick)")
        target_quat_0 = np.array([0, 0, 0, 1])
        try:
            obs, reward, done = rotate(env, task, target_quat=target_quat_0)
        except Exception as e:
            print(f"[Error] Exception during rotate to 0 deg: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after rotate to 0 deg!")
            shutdown_environment(env)
            return

        # Step 8: Move to item1 (ensure above for pick)
        print("[Step 8] Move to item1 (above for pick)")
        try:
            obs, reward, done = move(env, task, target_pos=item1_pos)
        except Exception as e:
            print(f"[Error] Exception during move to item1 (again): {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after move to item1 (again)!")
            shutdown_environment(env)
            return

        # Step 9: Pick item1 (first tomato)
        print("[Step 9] Pick item1 (first tomato)")
        try:
            obs, reward, done = pick(env, task, target_pos=item1_pos, approach_distance=0.10, approach_axis='z')
        except Exception as e:
            print(f"[Error] Exception during pick item1: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after picking item1!")
            shutdown_environment(env)
            return

        # Step 10: Place item1 on plate
        print("[Step 10] Place item1 on plate")
        try:
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.10, approach_axis='z')
        except Exception as e:
            print(f"[Error] Exception during place item1: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after placing item1!")
            shutdown_environment(env)
            return

        # Step 11: Move to item2 (second tomato)
        print("[Step 11] Move to item2 (second tomato)")
        try:
            obs, reward, done = move(env, task, target_pos=item2_pos)
        except Exception as e:
            print(f"[Error] Exception during move to item2: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after move to item2!")
            shutdown_environment(env)
            return

        # Step 12: Pick item2 (second tomato)
        print("[Step 12] Pick item2 (second tomato)")
        try:
            obs, reward, done = pick(env, task, target_pos=item2_pos, approach_distance=0.10, approach_axis='z')
        except Exception as e:
            print(f"[Error] Exception during pick item2: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after picking item2!")
            shutdown_environment(env)
            return

        # Step 13: Place item2 on plate
        print("[Step 13] Place item2 on plate")
        try:
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.10, approach_axis='z')
        except Exception as e:
            print(f"[Error] Exception during place item2: {e}")
            shutdown_environment(env)
            return
        if done:
            print("[Task] Task ended after placing item2!")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()