import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, then pick up all tomatoes and place them on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and environment
        # Drawer: choose 'bottom' drawer for this plan
        drawer_name = 'bottom'
        gripper_name = 'gripper'
        tomato_names = ['tomato1', 'tomato2']
        plate_name = 'plate'

        # Drawer handle/anchor/side positions
        # These keys must match those in get_object_positions()
        bottom_side_pos = positions.get('bottom_side_pos', None)
        bottom_anchor_pos = positions.get('bottom_anchor_pos', None)
        bottom_joint_pos = positions.get('bottom_joint_pos', None)
        # For robustness, fallback to alternative keys if needed
        if bottom_side_pos is None:
            bottom_side_pos = positions.get('side-pos-bottom', None)
        if bottom_anchor_pos is None:
            bottom_anchor_pos = positions.get('anchor-pos-bottom', None)
        if bottom_joint_pos is None:
            bottom_joint_pos = positions.get('joint-pos-bottom', None)
        # Plate position
        plate_pos = positions.get('plate', None)
        # Tomato positions
        tomato1_pos = positions.get('tomato1', None)
        tomato2_pos = positions.get('tomato2', None)

        # Step 1: Rotate gripper to 90 degrees (vertical for drawer manipulation)
        ninety_deg_quat = np.array([0, 0, np.sqrt(0.5), np.sqrt(0.5)])
        try:
            obs, reward, done = rotate(
                env,
                task,
                target_quat=ninety_deg_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Error] rotate failed: {e}")
            shutdown_environment(env)
            return

        # Step 2: Move to the side position of the bottom drawer
        if bottom_side_pos is None:
            print("[Error] bottom_side_pos not found in object positions.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(bottom_side_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after move to side pos!")
                return
        except Exception as e:
            print(f"[Error] move to side pos failed: {e}")
            shutdown_environment(env)
            return

        # Step 3: Move to the anchor position of the bottom drawer
        if bottom_anchor_pos is None:
            print("[Error] bottom_anchor_pos not found in object positions.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(bottom_anchor_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after move to anchor pos!")
                return
        except Exception as e:
            print(f"[Error] move to anchor pos failed: {e}")
            shutdown_environment(env)
            return

        # Step 4: Pick (grasp) the drawer handle at anchor position
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(bottom_anchor_pos),
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick drawer handle!")
                return
        except Exception as e:
            print(f"[Error] pick drawer handle failed: {e}")
            shutdown_environment(env)
            return

        # Step 5: Pull the drawer open (along x axis, positive direction)
        try:
            obs, reward, done = pull(
                env,
                task,
                pull_distance=0.20,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            shutdown_environment(env)
            return

        # Step 6: Pick up tomato1 from the table
        if tomato1_pos is None:
            print("[Error] tomato1 position not found.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(tomato1_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick tomato1!")
                return
        except Exception as e:
            print(f"[Error] pick tomato1 failed: {e}")
            shutdown_environment(env)
            return

        # Step 7: Place tomato1 on the plate
        if plate_pos is None:
            print("[Error] plate position not found.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=np.array(plate_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after place tomato1!")
                return
        except Exception as e:
            print(f"[Error] place tomato1 failed: {e}")
            shutdown_environment(env)
            return

        # Step 8: Pick up tomato2 from the table
        if tomato2_pos is None:
            print("[Error] tomato2 position not found.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(tomato2_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick tomato2!")
                return
        except Exception as e:
            print(f"[Error] pick tomato2 failed: {e}")
            shutdown_environment(env)
            return

        # Step 9: Place tomato2 on the plate
        if plate_pos is None:
            print("[Error] plate position not found.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=np.array(plate_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after place tomato2!")
                return
        except Exception as e:
            print(f"[Error] place tomato2 failed: {e}")
            shutdown_environment(env)
            return

        print("[Task] All steps completed successfully!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()