import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- PLAN EXECUTION STARTS HERE ---
        # Step 1: rotate gripper to 90 degrees (ninety_deg)
        try:
            from scipy.spatial.transform import Rotation as R
            obs = task.get_observation()
            ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()
            obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Step 1 (rotate) failed: {e}")
            shutdown_environment(env)
            return

        # Step 2: move gripper to side position of bottom drawer
        try:
            bottom_side_pos = positions.get('bottom_side_pos', None)
            if bottom_side_pos is None:
                raise KeyError("bottom_side_pos not found in object positions.")
            obs, reward, done = move(env, task, target_pos=np.array(bottom_side_pos))
            if done:
                print("[Task] Task ended after move to bottom_side_pos!")
                return
        except Exception as e:
            print(f"[ERROR] Step 2 (move to bottom_side_pos) failed: {e}")
            shutdown_environment(env)
            return

        # Step 3: move gripper to anchor position of bottom drawer
        try:
            bottom_anchor_pos = positions.get('bottom_anchor_pos', None)
            if bottom_anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in object positions.")
            obs, reward, done = move(env, task, target_pos=np.array(bottom_anchor_pos))
            if done:
                print("[Task] Task ended after move to bottom_anchor_pos!")
                return
        except Exception as e:
            print(f"[ERROR] Step 3 (move to bottom_anchor_pos) failed: {e}")
            shutdown_environment(env)
            return

        # Step 4: move gripper to bottom_joint_pos (simulate grasping the drawer handle)
        try:
            bottom_joint_pos = positions.get('bottom_joint_pos', None)
            if bottom_joint_pos is None:
                raise KeyError("bottom_joint_pos not found in object positions.")
            obs, reward, done = move(env, task, target_pos=np.array(bottom_joint_pos))
            if done:
                print("[Task] Task ended after move to bottom_joint_pos!")
                return
        except Exception as e:
            print(f"[ERROR] Step 4 (move to bottom_joint_pos) failed: {e}")
            shutdown_environment(env)
            return

        # Step 5: pull the drawer open
        try:
            pull_distance = 0.15  # meters, adjust as needed for your environment
            obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[ERROR] Step 5 (pull) failed: {e}")
            shutdown_environment(env)
            return

        # Step 6: pick up tomato1 from table
        try:
            tomato1_pos = positions.get('tomato1', None)
            if tomato1_pos is None:
                raise KeyError("tomato1 not found in object positions.")
            obs, reward, done = pick(env, task, target_pos=np.array(tomato1_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Step 6 (pick tomato1) failed: {e}")
            shutdown_environment(env)
            return

        # Step 7: place tomato1 on plate
        try:
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise KeyError("plate not found in object positions.")
            obs, reward, done = place(env, task, target_pos=np.array(plate_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Step 7 (place tomato1) failed: {e}")
            shutdown_environment(env)
            return

        # Step 8: pick up tomato2 from table
        try:
            tomato2_pos = positions.get('tomato2', None)
            if tomato2_pos is None:
                raise KeyError("tomato2 not found in object positions.")
            obs, reward, done = pick(env, task, target_pos=np.array(tomato2_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Step 8 (pick tomato2) failed: {e}")
            shutdown_environment(env)
            return

        # Step 9: place tomato2 on plate
        try:
            obs, reward, done = place(env, task, target_pos=np.array(plate_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place tomato2! Reward:", reward)
                return
        except Exception as e:
            print(f"[ERROR] Step 9 (place tomato2) failed: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()