import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task logic for: Unlock the cabinet by opening an accessible drawer, then transfer the 2 tomatoes onto the plate, leaving any other items undisturbed.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dict mapping object names to their 3D positions
        positions = get_object_positions()

        # --- Step 1: Rotate gripper to ninety_deg ---
        # We assume 'zero_deg' and 'ninety_deg' are known quaternions or can be constructed.
        # If not, we define them as standard orientations.
        # For RLBench, gripper orientation is usually quaternion [x, y, z, w].
        # Let's define zero_deg as identity quaternion, ninety_deg as 90deg about z axis.
        def get_quat_from_angle(angle_name):
            if angle_name == 'zero_deg':
                return np.array([0, 0, 0, 1])  # Identity quaternion
            elif angle_name == 'ninety_deg':
                # 90 deg about z axis: [axis * sin(theta/2), cos(theta/2)]
                theta = np.deg2rad(90)
                return np.array([0, 0, np.sin(theta/2), np.cos(theta/2)])
            else:
                raise ValueError(f"Unknown angle name: {angle_name}")

        # Step 1: rotate(gripper, zero_deg, ninety_deg)
        try:
            print("[Plan Step 1] Rotating gripper from zero_deg to ninety_deg")
            target_quat = get_quat_from_angle('ninety_deg')
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Step 1 (rotate) failed: {e}")
            return

        # --- Step 2: Move gripper to side-pos-bottom ---
        # The plan uses "move" with a single argument: "gripper", "side-pos-bottom"
        # We interpret this as: move gripper to the position of "side-pos-bottom"
        try:
            print("[Plan Step 2] Moving gripper to side-pos-bottom")
            target_pos = positions.get('bottom_side_pos') or positions.get('side-pos-bottom')
            if target_pos is None:
                raise KeyError("Position for 'side-pos-bottom' not found in object_positions.")
            obs, reward, done = move(env, task, np.array(target_pos))
            if done:
                print("[Task] Task ended after move to side-pos-bottom!")
                return
        except Exception as e:
            print(f"[ERROR] Step 2 (move to side-pos-bottom) failed: {e}")
            return

        # --- Step 3: Move gripper to anchor-pos-bottom ---
        try:
            print("[Plan Step 3] Moving gripper to anchor-pos-bottom")
            target_pos = positions.get('bottom_anchor_pos') or positions.get('anchor-pos-bottom')
            if target_pos is None:
                raise KeyError("Position for 'anchor-pos-bottom' not found in object_positions.")
            obs, reward, done = move(env, task, np.array(target_pos))
            if done:
                print("[Task] Task ended after move to anchor-pos-bottom!")
                return
        except Exception as e:
            print(f"[ERROR] Step 3 (move to anchor-pos-bottom) failed: {e}")
            return

        # --- Step 4: (Repeat) Move gripper to anchor-pos-bottom ---
        try:
            print("[Plan Step 4] Moving gripper to anchor-pos-bottom (repeat)")
            target_pos = positions.get('bottom_anchor_pos') or positions.get('anchor-pos-bottom')
            if target_pos is None:
                raise KeyError("Position for 'anchor-pos-bottom' not found in object_positions.")
            obs, reward, done = move(env, task, np.array(target_pos))
            if done:
                print("[Task] Task ended after repeated move to anchor-pos-bottom!")
                return
        except Exception as e:
            print(f"[ERROR] Step 4 (repeat move to anchor-pos-bottom) failed: {e}")
            return

        # --- Step 5: Pull to open the bottom drawer ---
        # The pull skill requires a distance and axis. We'll assume the drawer pulls along +x by 0.15m.
        try:
            print("[Plan Step 5] Pulling to open the bottom drawer")
            pull_distance = 0.15  # meters, typical drawer pull
            pull_axis = 'x'       # assume drawer opens along +x
            obs, reward, done = pull(env, task, pull_distance, pull_axis)
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[ERROR] Step 5 (pull) failed: {e}")
            return

        # --- Step 6: Pick up tomato1 from table ---
        try:
            print("[Plan Step 6] Picking up tomato1 from table")
            tomato1_pos = positions.get('tomato1')
            if tomato1_pos is None:
                raise KeyError("Position for 'tomato1' not found in object_positions.")
            obs, reward, done = pick(env, task, np.array(tomato1_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after picking tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Step 6 (pick tomato1) failed: {e}")
            return

        # --- Step 7: Place tomato1 on plate ---
        try:
            print("[Plan Step 7] Placing tomato1 on plate")
            plate_pos = positions.get('plate')
            if plate_pos is None:
                raise KeyError("Position for 'plate' not found in object_positions.")
            obs, reward, done = place(env, task, np.array(plate_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after placing tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Step 7 (place tomato1) failed: {e}")
            return

        # --- Step 8: Pick up tomato2 from table ---
        try:
            print("[Plan Step 8] Picking up tomato2 from table")
            tomato2_pos = positions.get('tomato2')
            if tomato2_pos is None:
                raise KeyError("Position for 'tomato2' not found in object_positions.")
            obs, reward, done = pick(env, task, np.array(tomato2_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after picking tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Step 8 (pick tomato2) failed: {e}")
            return

        # --- Step 9: Place tomato2 on plate ---
        try:
            print("[Plan Step 9] Placing tomato2 on plate")
            plate_pos = positions.get('plate')
            if plate_pos is None:
                raise KeyError("Position for 'plate' not found in object_positions.")
            obs, reward, done = place(env, task, np.array(plate_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after placing tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Step 9 (place tomato2) failed: {e}")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()