import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dict mapping object names to their 3D positions (np.ndarray)
        positions = get_object_positions()

        # --- PLAN EXECUTION (according to the provided plan steps) ---

        # Step 1: rotate gripper to ninety_deg
        from scipy.spatial.transform import Rotation as R
        obs = task.get_observation()
        gripper_pos = obs.gripper_pose[:3]
        gripper_quat = obs.gripper_pose[3:7]
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # [x, y, z, w]
        try:
            print("[Step 1] Rotating gripper to ninety_deg")
            obs, reward, done = rotate(
                env,
                task,
                target_quat=ninety_deg_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # Step 2: move gripper to side-pos-bottom
        try:
            side_pos_bottom = positions.get('bottom_side_pos', None)
            if side_pos_bottom is None:
                raise KeyError("bottom_side_pos not found in object positions.")
            print("[Step 2] Moving gripper to side-pos-bottom")
            obs, reward, done = move(
                env,
                task,
                target_pos=side_pos_bottom,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during move to side-pos-bottom!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to side-pos-bottom: {e}")
            shutdown_environment(env)
            return

        # Step 3: move gripper to anchor-pos-bottom
        try:
            anchor_pos_bottom = positions.get('bottom_anchor_pos', None)
            if anchor_pos_bottom is None:
                raise KeyError("bottom_anchor_pos not found in object positions.")
            print("[Step 3] Moving gripper to anchor-pos-bottom")
            obs, reward, done = move(
                env,
                task,
                target_pos=anchor_pos_bottom,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during move to anchor-pos-bottom!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to anchor-pos-bottom: {e}")
            shutdown_environment(env)
            return

        # Step 4: pick the bottom drawer handle at anchor-pos-bottom
        try:
            print("[Step 4] Picking (grasping) bottom drawer at anchor-pos-bottom")
            obs, reward, done = pick(
                env,
                task,
                target_pos=anchor_pos_bottom,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick of drawer!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick of drawer: {e}")
            shutdown_environment(env)
            return

        # Step 5: pull the drawer open
        try:
            print("[Step 5] Pulling the bottom drawer open")
            obs, reward, done = pull(
                env,
                task,
                pull_distance=0.15,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # Step 6: pick tomato1 from table
        try:
            tomato1_pos = positions.get('tomato1', None)
            if tomato1_pos is None:
                raise KeyError("tomato1 not found in object positions.")
            print("[Step 6] Picking tomato1 from table")
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick of tomato1!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick of tomato1: {e}")
            shutdown_environment(env)
            return

        # Step 7: place tomato1 on plate
        try:
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise KeyError("plate not found in object positions.")
            print("[Step 7] Placing tomato1 on plate")
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during place of tomato1!")
                return
        except Exception as e:
            print(f"[Error] Exception during place of tomato1: {e}")
            shutdown_environment(env)
            return

        # Step 8: pick tomato2 from table
        try:
            tomato2_pos = positions.get('tomato2', None)
            if tomato2_pos is None:
                raise KeyError("tomato2 not found in object positions.")
            print("[Step 8] Picking tomato2 from table")
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick of tomato2!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick of tomato2: {e}")
            shutdown_environment(env)
            return

        # Step 9: place tomato2 on plate
        try:
            print("[Step 9] Placing tomato2 on plate")
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during place of tomato2!")
                return
        except Exception as e:
            print(f"[Error] Exception during place of tomato2: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()