import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Unlock the cabinet by opening an accessible drawer, then transfer the 2 tomatoes onto the plate, leaving any other items undisturbed.

        # --- Step 1: Rotate gripper to 90 degrees (for grasping drawer handle) ---
        from scipy.spatial.transform import Rotation as R
        try:
            quat_ninety_z = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        except Exception as e:
            print(f"[Error] Failed to compute 90 degree quaternion: {e}")
            shutdown_environment(env)
            return

        print("[Plan] Rotating gripper to 90 degrees about z-axis.")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=quat_ninety_z,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # --- Step 2: Move gripper to the side position of the bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] Could not find 'bottom_side_pos' in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Moving to bottom drawer side position.")
        obs, reward, done = move(
            env,
            task,
            target_pos=bottom_side_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 3: Move to anchor position of the bottom drawer (handle) ---
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] Could not find 'bottom_anchor_pos' in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Moving to bottom drawer anchor position (handle).")
        obs, reward, done = move(
            env,
            task,
            target_pos=bottom_anchor_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # --- Step 4: Pick (grasp) the drawer handle ---
        print("[Plan] Grasping the bottom drawer handle.")
        obs, reward, done = pick(
            env,
            task,
            target_pos=bottom_anchor_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (drawer handle)!")
            return

        # --- Step 5: Pull the drawer open ---
        print("[Plan] Pulling the bottom drawer open.")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.15,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull (drawer)!")
            return

        # --- Step 6: Move gripper above item1 (tomato1) on the table ---
        try:
            item1_pos = positions['item1']
        except KeyError:
            print("[Error] Could not find 'item1' in object positions.")
            shutdown_environment(env)
            return

        item1_above = np.array(item1_pos) + np.array([0, 0, 0.10])
        print("[Plan] Moving above item1 (tomato1).")
        obs, reward, done = move(
            env,
            task,
            target_pos=item1_above,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move above item1!")
            return

        # --- Step 7: Pick item1 (tomato1) ---
        print("[Plan] Picking up item1 (tomato1).")
        obs, reward, done = pick(
            env,
            task,
            target_pos=item1_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (item1)!")
            return

        # --- Step 8: Move above the plate ---
        try:
            plate_pos = positions['plate']
        except KeyError:
            print("[Error] Could not find 'plate' in object positions.")
            shutdown_environment(env)
            return

        plate_above = np.array(plate_pos) + np.array([0, 0, 0.10])
        print("[Plan] Moving above the plate.")
        obs, reward, done = move(
            env,
            task,
            target_pos=plate_above,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move above plate!")
            return

        # --- Step 9: Place item1 (tomato1) on the plate ---
        print("[Plan] Placing item1 (tomato1) on the plate.")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place (item1)!")
            return

        # --- Step 10: Move above item2 (tomato2) on the table ---
        try:
            item2_pos = positions['item2']
        except KeyError:
            print("[Error] Could not find 'item2' in object positions.")
            shutdown_environment(env)
            return

        item2_above = np.array(item2_pos) + np.array([0, 0, 0.10])
        print("[Plan] Moving above item2 (tomato2).")
        obs, reward, done = move(
            env,
            task,
            target_pos=item2_above,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move above item2!")
            return

        # --- Step 11: Pick item2 (tomato2) ---
        print("[Plan] Picking up item2 (tomato2).")
        obs, reward, done = pick(
            env,
            task,
            target_pos=item2_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick (item2)!")
            return

        # --- Step 12: Move above the plate (again) ---
        print("[Plan] Moving above the plate (for item2).")
        obs, reward, done = move(
            env,
            task,
            target_pos=plate_above,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move above plate (item2)!")
            return

        # --- Step 13: Place item2 (tomato2) on the plate ---
        print("[Plan] Placing item2 (tomato2) on the plate.")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place (item2)!")
            return

        print("[Task] Task completed successfully! Both tomatoes placed on the plate and drawer opened.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()