import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open one of the drawers in the cabinet, then put all tomatoes on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and observation:
        # Drawers: 'bottom', 'middle', 'top'
        # Drawer positions: e.g., 'bottom_anchor_pos', 'bottom_side_pos', etc.
        # Items: 'item1', 'item2', 'item3' (assume these are tomatoes)
        # Plate: 'plate'

        # For robustness, check for required keys
        required_keys = [
            'bottom_anchor_pos', 'bottom_side_pos', 'bottom_joint_pos',
            'middle_anchor_pos', 'middle_side_pos', 'middle_joint_pos',
            'top_anchor_pos', 'top_side_pos', 'top_joint_pos',
            'item1', 'item2', 'item3', 'plate'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Missing object position for: {k}")

        # Choose a drawer to open (bottom drawer for simplicity)
        drawer_name = 'bottom'
        gripper_name = 'gripper'
        anchor_pos = positions['bottom_anchor_pos']
        side_pos = positions['bottom_side_pos']
        joint_pos = positions['bottom_joint_pos']

        # Items to move (tomatoes)
        tomato_names = ['item1', 'item2', 'item3']
        tomato_positions = [positions[name] for name in tomato_names]
        plate_pos = positions['plate']

        # Precompute required quaternions and pull distance for the plan
        from scipy.spatial.transform import Rotation as R
        # 90 deg rotation about z for drawer grasp
        quat_90_z = R.from_euler('z', 90, degrees=True).as_quat()
        # 0 deg rotation about z for tomato pick
        quat_0 = R.from_euler('z', 0, degrees=True).as_quat()
        # Pull distance for opening drawer (tuned for environment)
        pull_distance = 0.18

        # For above-tomato positions
        tomato1_pos = positions['item1']
        tomato1_above = np.array(tomato1_pos) + np.array([0, 0, 0.10])
        tomato2_pos = positions['item2']
        tomato2_above = np.array(tomato2_pos) + np.array([0, 0, 0.10])
        tomato3_pos = positions['item3']
        tomato3_above = np.array(tomato3_pos) + np.array([0, 0, 0.10])

        # === PLAN EXECUTION ===
        # [Frozen Code Start]
        obs, reward, done = move(
                env, task,
                target_pos=side_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        obs, reward, done = rotate(
                env, task,
                target_quat=quat_90_z,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
        obs, reward, done = move(
                env, task,
                target_pos=anchor_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        obs, reward, done = pick(
                env, task,
                target_pos=anchor_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        obs, reward, done = pull(
                env, task,
                pull_distance=pull_distance,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        obs, reward, done = move(
                env, task,
                target_pos=tomato1_above,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        obs, reward, done = rotate(
                env, task,
                target_quat=quat_0,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
        obs, reward, done = pick(
                env, task,
                target_pos=tomato1_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        obs, reward, done = place(
                env, task,
                target_pos=plate_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        obs, reward, done = move(
                    env, task,
                    target_pos=tomato2_above,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
        obs, reward, done = pick(
                    env, task,
                    target_pos=tomato2_pos,
                    approach_distance=0.10,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
        obs, reward, done = place(
                    env, task,
                    target_pos=plate_pos,
                    approach_distance=0.10,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
        # [Frozen Code End]

        # === Continue with remaining tomatoes ===

        # Step 13: Move above third tomato
        print("[Step 13] Move gripper above third tomato")
        obs, reward, done = move(
            env, task,
            target_pos=tomato3_above,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move above tomato3!")
            shutdown_environment(env)
            return

        # Step 14: Pick third tomato
        print("[Step 14] Pick third tomato")
        obs, reward, done = pick(
            env, task,
            target_pos=tomato3_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick tomato3!")
            shutdown_environment(env)
            return

        # Step 15: Place third tomato on plate
        print("[Step 15] Place third tomato on plate")
        obs, reward, done = place(
            env, task,
            target_pos=plate_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place tomato3!")
            shutdown_environment(env)
            return

        print("[Task] All tomatoes placed on plate. Task completed successfully! Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()