import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open one of the drawers in the cabinet, then put all tomatoes on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys (from object list): 
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1',
        # 'rubbish', 'tomato1', 'tomato2', 'plate'

        # --- Choose which drawer to open (bottom) ---
        drawer = 'bottom'
        gripper_name = 'gripper'
        # Positions for bottom drawer
        side_pos = positions.get('bottom_side_pos', None)
        anchor_pos = positions.get('bottom_anchor_pos', None)
        joint_pos = positions.get('bottom_joint_pos', None)
        # Plate position
        plate_pos = positions.get('plate', None)
        # Tomato positions
        tomato1_pos = positions.get('tomato1', None)
        tomato2_pos = positions.get('tomato2', None)

        # === Plan Execution ===
        # Step 1: rotate (rotate gripper to 90 degrees for side approach)
        # Assume 90 deg about z axis: quaternion for 90 deg (z) is [0, 0, sqrt(0.5), sqrt(0.5)]
        target_quat_ninety_z = np.array([0, 0, np.sqrt(0.5), np.sqrt(0.5)])
        [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat=target_quat_ninety_z)
        obs, reward, done = move(env, task, target_pos=side_pos)
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        obs, reward, done = pick(env, task, target_pos=tomato1_pos, approach_distance=0.15, approach_axis='z')
        [Frozen Code End]
        if done:
            print("[Task] Task ended after initial steps!")
            return

        # Step 7: place tomato1 on plate
        try:
            if plate_pos is None:
                raise ValueError("plate not found in positions.")
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place tomato1!")
                return
        except Exception as e:
            print(f"[Error] place tomato1 failed: {e}")
            return

        # Step 8: pick tomato2 from table
        try:
            if tomato2_pos is None:
                raise ValueError("tomato2 not found in positions.")
            obs, reward, done = pick(env, task, target_pos=tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick tomato2!")
                return
        except Exception as e:
            print(f"[Error] pick tomato2 failed: {e}")
            return

        # Step 9: place tomato2 on plate
        try:
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place tomato2!")
                return
        except Exception as e:
            print(f"[Error] place tomato2 failed: {e}")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()