# ====== (No code block, plain text as per instructions) ======

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Pull open any drawer that is not locked, then drop the 2 tomatoes onto the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and observation
        # Drawers: bottom, middle, top
        # Tomatoes: tomato1, tomato2
        # Plate: plate

        # 1. Find an unlocked drawer (bottom is not locked in initial state)
        unlocked_drawer = 'bottom'
        drawer_anchor_pos_name = 'bottom_anchor_pos'
        drawer_joint_pos_name = 'bottom_joint_pos'
        drawer_side_pos_name = 'bottom_side_pos'

        # 2. Get positions for manipulation
        try:
            anchor_pos = positions[drawer_anchor_pos_name]
            joint_pos = positions[drawer_joint_pos_name]
            side_pos = positions[drawer_side_pos_name]
        except KeyError as e:
            print(f"[Error] Could not find drawer position: {e}")
            shutdown_environment(env)
            return

        # 3. Get tomato and plate positions
        try:
            tomato1_pos = positions['tomato1']
            tomato2_pos = positions['tomato2']
            plate_pos = positions['plate']
        except KeyError as e:
            print(f"[Error] Could not find object position: {e}")
            shutdown_environment(env)
            return

        # === PLAN EXECUTION ===

        # Step 1: Rotate gripper to 90 degrees (for side approach)
        # Assume 90 deg quaternion is [0, 0, sqrt(0.5), sqrt(0.5)] (z axis, 90 deg)
        # But let's get current orientation and rotate around z by 90 deg
        obs = task.get_observation()
        current_quat = obs.gripper_pose[3:7]
        from scipy.spatial.transform import Rotation as R
        rot = R.from_quat(current_quat)
        rot_90 = R.from_euler('z', 90, degrees=True)
        target_quat = (rot * rot_90).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Task ended during rotate!")
            return

        # Step 2: Move to side position of the drawer
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Task ended during move to side position!")
            return

        # Step 3: Move to anchor position (to grab the drawer handle)
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Task ended during move to anchor position!")
            return

        # Step 4: Move slightly into the drawer handle (simulate grasping)
        # We'll use the joint position as a point slightly "inside" the drawer
        obs, reward, done = move(env, task, joint_pos)
        if done:
            print("[Task] Task ended during move to joint position!")
            return

        # Step 5: Pull the drawer open (pull along x axis, positive direction)
        # Estimate pull distance (e.g., 0.15m)
        pull_distance = 0.15
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended during pull!")
            return

        # Step 6: Pick up tomato1 from table
        obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended during pick tomato1!")
            return

        # Step 7: Place tomato1 on plate
        obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended during place tomato1!")
            return

        # Step 8: Pick up tomato2 from table
        obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended during pick tomato2!")
            return

        # Step 9: Place tomato2 on plate
        obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended during place tomato2!")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
