import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and environment:
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'waypoint1', 'rubbish', 'tomato1', 'tomato2', 'plate'

        # === Plan Execution ===
        # Goal: Select a drawer and open it fully, then pick up all the tomatoes and leave them on the plate.

        # --- Step 1: Rotate gripper to 90 degrees (for side approach) ---
        try:
            # Get current gripper pose
            obs = task.get_observation()
            start_quat = obs.gripper_pose[3:7]
            # Target: 90 deg about z axis (assuming 'zero_deg' is 0, 'ninety_deg' is 90 deg about z)
            # We'll use scipy to get the quaternion for 90 deg about z
            from scipy.spatial.transform import Rotation as R
            target_rot = R.from_euler('z', 90, degrees=True)
            target_quat = target_rot.as_quat()  # xyzw
            [Frozen Code Start]
            obs, reward, done = rotate(env, task, target_quat)
            [Frozen Code End]
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move to side position of bottom drawer ---
        try:
            bottom_side_pos = positions.get('bottom_side_pos', None)
            if bottom_side_pos is None:
                raise ValueError("bottom_side_pos not found in positions.")
            [Frozen Code Start]
            obs, reward, done = move(env, task, bottom_side_pos)
            [Frozen Code End]
            if done:
                print("[Task] Task ended during move to bottom_side_pos!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to bottom_side_pos: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        try:
            bottom_anchor_pos = positions.get('bottom_anchor_pos', None)
            if bottom_anchor_pos is None:
                raise ValueError("bottom_anchor_pos not found in positions.")
            [Frozen Code Start]
            obs, reward, done = move(env, task, bottom_anchor_pos)
            [Frozen Code End]
            if done:
                print("[Task] Task ended during move to bottom_anchor_pos!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to bottom_anchor_pos: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: Pick the drawer handle (simulate pick-drawer by pick at anchor pos) ---
        try:
            # For the drawer, we simulate "pick" by closing the gripper at the anchor position
            [Frozen Code Start]
            obs, reward, done = pick(env, task, bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
            [Frozen Code End]
            if done:
                print("[Task] Task ended during pick-drawer!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick-drawer: {e}")
            shutdown_environment(env)
            return

        # --- Step 5: Pull the drawer open (pull along -x axis, e.g., 0.20m) ---
        try:
            pull_distance = 0.20  # meters, adjust as needed for full open
            [Frozen Code Start]
            obs, reward, done = pull(env, task, pull_distance, pull_axis='-x')
            [Frozen Code End]
            if done:
                print("[Task] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Pick up tomato1 from table ---
        try:
            tomato1_pos = positions.get('tomato1', None)
            if tomato1_pos is None:
                raise ValueError("tomato1 not found in positions.")
            [Frozen Code Start]
            obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15, approach_axis='z')
            [Frozen Code End]
            if done:
                print("[Task] Task ended during pick tomato1!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick tomato1: {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place tomato1 on plate ---
        try:
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise ValueError("plate not found in positions.")
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place tomato1!")
                return
        except Exception as e:
            print(f"[Error] Exception during place tomato1: {e}")
            shutdown_environment(env)
            return

        # --- Step 8: Pick up tomato2 from table ---
        try:
            tomato2_pos = positions.get('tomato2', None)
            if tomato2_pos is None:
                raise ValueError("tomato2 not found in positions.")
            obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick tomato2!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick tomato2: {e}")
            shutdown_environment(env)
            return

        # --- Step 9: Place tomato2 on plate ---
        try:
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place tomato2!")
                return
        except Exception as e:
            print(f"[Error] Exception during place tomato2: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()