import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Plan Execution ===

        # Step 1: rotate (rotate gripper to 90 degrees to prepare for drawer manipulation)
        try:
            obs = task.get_observation()
            start_quat = obs.gripper_pose[3:7]
            from scipy.spatial.transform import Rotation as R
            target_rot = R.from_euler('z', 90, degrees=True)
            target_quat = target_rot.as_quat()  # xyzw
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # Step 2: move (move gripper to side position of bottom drawer)
        try:
            if 'bottom_side_pos' in positions:
                side_pos = positions['bottom_side_pos']
            elif 'side-pos-bottom' in positions:
                side_pos = positions['side-pos-bottom']
            else:
                raise KeyError("bottom_side_pos not found in positions.")
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Task] Task ended during move to side position!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to side position: {e}")
            shutdown_environment(env)
            return

        # Step 3: move (move gripper to anchor position of bottom drawer)
        try:
            if 'bottom_anchor_pos' in positions:
                anchor_pos = positions['bottom_anchor_pos']
            elif 'anchor-pos-bottom' in positions:
                anchor_pos = positions['anchor-pos-bottom']
            else:
                raise KeyError("bottom_anchor_pos not found in positions.")
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Task ended during move to anchor position!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to anchor position: {e}")
            shutdown_environment(env)
            return

        # Step 4: pick (pick the drawer handle at anchor position)
        try:
            obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended during pick drawer handle!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick drawer handle: {e}")
            shutdown_environment(env)
            return

        # Step 5: pull (pull the drawer open)
        try:
            pull_distance = 0.15
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended during pull drawer!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull drawer: {e}")
            shutdown_environment(env)
            return

        # Step 6: pick tomato1 from table
        try:
            if 'tomato1' not in positions:
                raise KeyError("tomato1 not found in positions.")
            tomato1_pos = positions['tomato1']
            obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick tomato1!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick tomato1: {e}")
            shutdown_environment(env)
            return

        # Step 7: place tomato1 on plate
        try:
            if 'plate' not in positions:
                raise KeyError("plate not found in positions.")
            plate_pos = positions['plate']
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place tomato1!")
                return
        except Exception as e:
            print(f"[Error] Exception during place tomato1: {e}")
            shutdown_environment(env)
            return

        # Step 8: pick tomato2 from table
        try:
            if 'tomato2' not in positions:
                raise KeyError("tomato2 not found in positions.")
            tomato2_pos = positions['tomato2']
            obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick tomato2!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick tomato2: {e}")
            shutdown_environment(env)
            return

        # Step 9: place tomato2 on plate
        try:
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place tomato2!")
                return
        except Exception as e:
            print(f"[Error] Exception during place tomato2: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()