import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open one of the drawers in the cabinet, then put all tomatoes on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and environment
        # Drawers: 'bottom', 'middle', 'top'
        # Tomatoes: 'tomato1', 'tomato2'
        # Plate: 'plate'
        # Drawer anchor/side/joint positions: e.g., 'bottom_anchor_pos', etc.

        # --- Step 1: Rotate gripper to 90 degrees (vertical for drawer manipulation) ---
        # Get current gripper pose
        obs = task.get_observation()
        start_quat = obs.gripper_pose[3:7]
        # Target orientation: 90 deg about z axis (assuming 'ninety_deg' means vertical)
        # We'll use the orientation of the anchor/joint of the drawer as reference if available
        # Otherwise, rotate 90 deg about z
        # For RLBench, let's use a quaternion for 90 deg about z
        from scipy.spatial.transform import Rotation as R
        target_rot = R.from_euler('z', 90, degrees=True)
        target_quat = target_rot.as_quat()  # xyzw
        # RLBench expects xyzw
        try:
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Task] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move to side position of a drawer (choose 'bottom' drawer for simplicity) ---
        # Use 'bottom_side_pos' as the approach position for the gripper
        try:
            side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Task] Could not find 'bottom_side_pos' in positions.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Task] Task ended during move to side position!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to side position: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move to anchor position of the drawer (for grasping handle) ---
        try:
            anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Task] Could not find 'bottom_anchor_pos' in positions.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Task ended during move to anchor position!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to anchor position: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: Pick the drawer handle (simulate grasp) ---
        # For the pick skill, we use the anchor position as the target
        try:
            obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended during pick drawer handle!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick drawer handle: {e}")
            shutdown_environment(env)
            return

        # --- Step 5: Pull the drawer open ---
        # Pull along the x axis (assume drawer opens along +x, adjust if needed)
        # Use a reasonable pull distance (e.g., 0.15 meters)
        try:
            obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
            if done:
                print("[Task] Task ended during pull (open drawer)!")
                return
        except Exception as e:
            print(f"[Task] Exception during pull (open drawer): {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Pick up tomato1 from table ---
        try:
            tomato1_pos = positions['tomato1']
        except KeyError:
            print("[Task] Could not find 'tomato1' in positions.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick tomato1!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick tomato1: {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place tomato1 on plate ---
        try:
            plate_pos = positions['plate']
        except KeyError:
            print("[Task] Could not find 'plate' in positions.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place tomato1!")
                return
        except Exception as e:
            print(f"[Task] Exception during place tomato1: {e}")
            shutdown_environment(env)
            return

        # --- Step 8: Pick up tomato2 from table ---
        try:
            tomato2_pos = positions['tomato2']
        except KeyError:
            print("[Task] Could not find 'tomato2' in positions.")
            shutdown_environment(env)
            return
        try:
            obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick tomato2!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick tomato2: {e}")
            shutdown_environment(env)
            return

        # --- Step 9: Place tomato2 on plate ---
        try:
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place tomato2!")
                return
        except Exception as e:
            print(f"[Task] Exception during place tomato2: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()