import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, then pick up all tomatoes and place them on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys: 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos', 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos', 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1', 'rubbish', 'tomato1', 'tomato2', 'plate'

        # --- PLAN: Open a drawer fully, then pick up all tomatoes and place them on the plate ---

        # 1. Select a drawer: We'll use the 'bottom' drawer for this plan.
        drawer_name = 'bottom'
        gripper_name = 'gripper'
        # Positions for the bottom drawer
        anchor_pos = positions.get('bottom_anchor_pos')
        side_pos = positions.get('bottom_side_pos')
        joint_pos = positions.get('bottom_joint_pos')
        # For pulling, we will move along the x-axis (assumption: drawer opens along x)
        pull_axis = 'x'
        pull_distance = 0.18  # Adjust as needed for "fully open" (domain knowledge)
        # For rotation, we assume 'ninety_deg' is [0, 0, 0.7071, 0.7071] in quaternion (90 deg about z)
        # If not available, use euler to quat conversion
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw

        # === [Frozen Code Start] ===
        obs, reward, done = rotate(
                env,
                task,
                target_quat=ninety_deg_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
        obs, reward, done = move(
                env,
                task,
                target_pos=side_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        obs, reward, done = move(
                env,
                task,
                target_pos=anchor_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        obs, reward, done = pull(
                env,
                task,
                pull_distance=pull_distance,
                pull_axis=pull_axis,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        obs, reward, done = pick(
                env,
                task,
                target_pos=positions.get('tomato1'),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        obs, reward, done = place(
                env,
                task,
                target_pos=positions.get('plate'),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        # === [Frozen Code End] ===

        # 8. [pick] Pick up tomato2 from the table
        tomato2_pos = positions.get('tomato2')
        if tomato2_pos is None:
            print("[Task] Error: tomato2 position not found in positions dictionary.")
            return
        try:
            print("[Task] Picking up tomato2 at:", tomato2_pos)
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after picking tomato2!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick tomato2: {e}")
            return

        # 9. [place] Place tomato2 on the plate
        plate_pos = positions.get('plate')
        if plate_pos is None:
            print("[Task] Error: plate position not found in positions dictionary.")
            return
        try:
            print("[Task] Placing tomato2 on plate at:", plate_pos)
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after placing tomato2!")
                return
        except Exception as e:
            print(f"[Task] Exception during place tomato2: {e}")
            return

        print("[Task] All tomatoes placed on plate. Task completed successfully!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()