# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example: positions = {'drawer': (x, y, z), 'object_1': (x, y, z), ...}

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will use available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We'll try to use each skill and observe if any information is revealed or if any action fails due to missing knowledge.

        # 1. Try to move to each location (if move is implemented as a skill)
        #    - In this domain, 'move' is in available skills, but its parameters are not fully specified.
        #    - If move is not implemented, skip to pick/place/rotate/pull.

        # 2. Try to pick up each object and see if any knowledge is revealed (e.g., weight-known, durability-known).
        # 3. Try to pull drawers and see if lock-known is revealed.

        # For demonstration, we will attempt to pick and pull on all objects and drawers.

        # --- Exploration: Try picking up all objects ---
        for obj_name, obj_pos in positions.items():
            try:
                print(f"[Exploration] Attempting to pick object: {obj_name} at {obj_pos}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Could not pick {obj_name}: {e}")

        # --- Exploration: Try pulling all drawers (if any) ---
        for obj_name, obj_pos in positions.items():
            if "drawer" in obj_name.lower():
                try:
                    print(f"[Exploration] Attempting to pull drawer: {obj_name} at {obj_pos}")
                    obs, reward, done = pull(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after pulling {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Could not pull {obj_name}: {e}")

        # --- Exploration: Try rotating gripper at each object (if relevant) ---
        for obj_name, obj_pos in positions.items():
            try:
                print(f"[Exploration] Attempting to rotate at {obj_name} (if applicable)")
                # Example: rotate to a fixed quaternion (identity)
                target_quat = np.array([0, 0, 0, 1])
                obs, reward, done = rotate(
                    env,
                    task,
                    target_quat=target_quat,
                    max_steps=50,
                    threshold=0.05,
                    timeout=5.0
                )
                if done:
                    print(f"[Exploration] Task ended after rotating at {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Could not rotate at {obj_name}: {e}")

        # --- Exploration: Try placing any held object at its original position ---
        for obj_name, obj_pos in positions.items():
            try:
                print(f"[Exploration] Attempting to place object: {obj_name} at {obj_pos}")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after placing {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Could not place {obj_name}: {e}")

        print("[Exploration] Completed exploration phase. If actions failed, check logs for missing predicate clues.")

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this code, we focus on the exploration phase as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
