# run_skeleton_task.py (Completed with Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate ===
        # The feedback and exploration domain suggest that some knowledge (e.g., lock-known, weight-known, durability-known, temperature-known, identified) may be missing.
        # We will attempt to discover which predicate is missing by performing available skills and observing the effects.

        # For demonstration, let's try to pick and pull an object (e.g., a drawer handle) and see if we can discover the lock state.
        # We'll use only the available skills: pick, place, move, rotate, pull

        # 1. Identify a candidate object for exploration (e.g., a drawer or handle)
        #    We'll use the first object in positions as a candidate.
        candidate_obj = None
        candidate_loc = None
        for obj_name, obj_info in positions.items():
            # Assume obj_info is a dict with at least 'type' and 'position'
            if 'drawer' in obj_name or 'handle' in obj_name or 'drawer' in obj_info.get('type', ''):
                candidate_obj = obj_name
                candidate_loc = obj_info.get('location', None)
                break
        if candidate_obj is None:
            # Fallback: just pick the first object
            for obj_name, obj_info in positions.items():
                candidate_obj = obj_name
                candidate_loc = obj_info.get('location', None)
                break

        print(f"[Exploration] Candidate object for exploration: {candidate_obj} at {candidate_loc}")

        # 2. Try to pick the object (if possible)
        try:
            print(f"[Exploration] Attempting to pick {candidate_obj}...")
            obs, reward, done = pick(
                env,
                task,
                target_obj=candidate_obj,
                target_pos=positions[candidate_obj]['position'],
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")

        # 3. Try to pull the object (if possible)
        try:
            print(f"[Exploration] Attempting to pull {candidate_obj}...")
            obs, reward, done = pull(
                env,
                task,
                target_obj=candidate_obj,
                target_pos=positions[candidate_obj]['position'],
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")

        # 4. Try to rotate the gripper (if possible)
        try:
            print(f"[Exploration] Attempting to rotate gripper for {candidate_obj}...")
            # For demonstration, rotate to a fixed quaternion (identity)
            target_quat = np.array([0, 0, 0, 1])
            obs, reward, done = rotate(
                env,
                task,
                target_quat=target_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate: {e}")

        # 5. Try to place the object (if possible)
        try:
            print(f"[Exploration] Attempting to place {candidate_obj}...")
            # Place at the original position for simplicity
            obs, reward, done = place(
                env,
                task,
                target_obj=candidate_obj,
                target_pos=positions[candidate_obj]['position'],
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended after place!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during place: {e}")

        # 6. Try to move (if possible)
        try:
            print(f"[Exploration] Attempting to move to another location...")
            # Find another location to move to
            other_loc = None
            for obj_name, obj_info in positions.items():
                loc = obj_info.get('location', None)
                if loc and loc != candidate_loc:
                    other_loc = loc
                    break
            if other_loc:
                obs, reward, done = move(
                    env,
                    task,
                    from_location=candidate_loc,
                    to_location=other_loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after move!")
                    return
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")

        print("[Exploration] Exploration phase complete. Check logs and environment state for discovered predicates.")

        # === End of Exploration Phase ===

        # At this point, based on the feedback and exploration, you should be able to determine which predicate is missing
        # (e.g., lock-known, weight-known, durability-known, temperature-known, identified).
        # This can be used to inform further planning or debugging.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
