# run_skeleton_task.py (Completed with Exploration Phase and Predicate Check)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Check for Missing Predicate (e.g., Drawer Lock State) ===
        # The feedback indicates we must check if a drawer is locked before attempting to rotate or pull it.
        # We'll try to infer the lock state by attempting a pull and observing the result, or by checking any available state.

        # For demonstration, let's assume the drawer is named 'drawer_bottom' and the gripper is 'gripper_1'
        # These names should be replaced with actual object names from your environment if different.
        drawer_name = 'drawer_bottom'
        gripper_name = 'gripper_1'
        anchor_pos_name = 'anchor_pos_bottom'
        side_pos_name = 'side_pos_bottom'
        angle_ninety = 'ninety_deg'
        angle_zero = 'zero_deg'

        # Get positions for the drawer and gripper anchor/side positions if available
        drawer_pos = positions.get(drawer_name, None)
        anchor_pos = positions.get(anchor_pos_name, None)
        side_pos = positions.get(side_pos_name, None)

        # Step 1: Exploration - Try to determine if the drawer is locked
        print("[Exploration] Checking if the drawer is locked before manipulation...")

        is_locked = False
        try:
            # Try to pull the drawer to see if it is locked
            # This is a proxy for "exploring" the lock state
            obs, reward, done = pull(env, task, gripper_name, drawer_name)
            # If the drawer did not open, we assume it is locked
            # (In a real environment, you would check a predicate or observation)
            if hasattr(obs, 'is_locked'):
                is_locked = obs.is_locked
            else:
                # Fallback: If the drawer did not open, treat as locked
                if not getattr(obs, 'drawer_open', False):
                    is_locked = True
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")
            is_locked = True

        if is_locked:
            print("[Exploration] Drawer is locked. Skipping rotation and pull actions.")
            # Optionally, you could return or handle this case differently
        else:
            print("[Exploration] Drawer is unlocked. Proceeding with manipulation.")

            # Step 2: Rotate gripper to 90 degrees (if required)
            try:
                # For demonstration, assume we have a quaternion for 90 degrees
                # Replace with actual quaternion if available
                target_quat = np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)])  # 90 deg about z
                obs, reward, done = rotate(env, task, target_quat)
                if done:
                    print("[Task] Task ended during rotation!")
                    return
            except Exception as e:
                print(f"[Task] Exception during rotate: {e}")

            # Step 3: Move gripper to side position (if required)
            try:
                # If a move skill is available and needed, call it here
                # move(env, task, from_pos, to_pos)
                pass  # Not implemented in provided skills
            except Exception as e:
                print(f"[Task] Exception during move-to-side: {e}")

            # Step 4: Move gripper to anchor position (if required)
            try:
                # If a move skill is available and needed, call it here
                # move(env, task, from_pos, to_pos)
                pass  # Not implemented in provided skills
            except Exception as e:
                print(f"[Task] Exception during move-to-anchor: {e}")

            # Step 5: Pull the drawer open
            try:
                obs, reward, done = pull(env, task, gripper_name, drawer_name)
                if done:
                    print("[Task] Task ended after pulling drawer!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pull: {e}")

        # === Continue with the rest of the plan (pick/place etc.) as needed ===
        # Example: Pick an object from the drawer and place it somewhere
        object_to_pick = 'object_1'
        pick_location = 'drawer_bottom'
        place_location = 'table_top'

        try:
            print(f"[Task] Attempting to pick {object_to_pick} from {pick_location}")
            obs, reward, done = pick(env, task, object_to_pick, pick_location)
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick: {e}")

        try:
            print(f"[Task] Attempting to place {object_to_pick} at {place_location}")
            obs, reward, done = place(env, task, object_to_pick, place_location)
            if done:
                print("[Task] Task ended after place!")
                return
        except Exception as e:
            print(f"[Task] Exception during place: {e}")

        print("[Task] Plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()