# run_skeleton_task.py (Completed with Exploration Phase and Feedback Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The goal is to discover which predicate is missing (e.g., lock-known, identified, etc.)
        # We will attempt to use available skills to interact with objects and observe the environment.
        # This phase is guided by the exploration knowledge and feedback.

        # Safety: Limit the number of exploration steps to avoid infinite loops
        MAX_EXPLORATION_STEPS = 10
        exploration_steps = 0

        # Example: Try to pick each object and see if any new information is revealed
        # (Assume positions is a dict: {object_name: (x, y, z), ...})
        # We'll also check for gripper and object validity before acting

        for obj_name, obj_pos in positions.items():
            if exploration_steps >= MAX_EXPLORATION_STEPS:
                print("[Exploration] Reached max exploration steps.")
                break

            print(f"[Exploration] Attempting to pick object: {obj_name} at {obj_pos}")

            try:
                # Safety check: Ensure object position is valid
                if obj_pos is None or not isinstance(obj_pos, (list, tuple)) or len(obj_pos) != 3:
                    print(f"[Exploration] Invalid position for {obj_name}, skipping.")
                    continue

                # Try to pick the object using the predefined pick skill
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=50,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )

                exploration_steps += 1

                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}.")
                    break

                # After pick, try to place the object back (if place skill is available)
                # For exploration, we can place it at the same position
                if 'place' in globals():
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=50,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=5.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after placing {obj_name}.")
                        break

            except Exception as e:
                print(f"[Exploration] Exception during pick/place of {obj_name}: {e}")
                continue

        # === Exploration: Try to rotate and pull drawers if present ===
        # This can help discover predicates like is-locked, is-open, etc.
        for obj_name, obj_pos in positions.items():
            if exploration_steps >= MAX_EXPLORATION_STEPS:
                print("[Exploration] Reached max exploration steps.")
                break

            if 'drawer' in obj_name.lower():
                print(f"[Exploration] Attempting to rotate and pull drawer: {obj_name}")

                try:
                    # Safety: Check if rotate skill is available
                    if 'rotate' in globals():
                        # Example: Try to rotate to 90 degrees (assuming quaternion for 90 deg about z)
                        target_quat = [0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]  # [x, y, z, w]
                        obs, reward, done = rotate(
                            env,
                            task,
                            target_quat=target_quat,
                            max_steps=50,
                            threshold=0.05,
                            timeout=5.0
                        )
                        exploration_steps += 1
                        if done:
                            print(f"[Exploration] Task ended after rotating {obj_name}.")
                            break

                    # Try to pull the drawer if possible
                    if 'pull' in globals():
                        obs, reward, done = pull(
                            env,
                            task,
                            drawer_name=obj_name
                        )
                        exploration_steps += 1
                        if done:
                            print(f"[Exploration] Task ended after pulling {obj_name}.")
                            break

                except Exception as e:
                    print(f"[Exploration] Exception during rotate/pull of {obj_name}: {e}")
                    continue

        print("[Exploration] Exploration phase complete. If new predicates were discovered, update your plan accordingly.")

        # === Main Task Plan (Oracle Plan Execution) ===
        # After exploration, proceed with the main plan using only available skills.
        # Example: For each object, pick and place at a target location (if required by the task).
        # This is a placeholder; actual plan should be inserted here.

        for obj_name, obj_pos in positions.items():
            print(f"[Task] Manipulating object: {obj_name} at {obj_pos}")

            try:
                # Pick the object
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=50,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
                if done:
                    print(f"[Task] Task ended after picking {obj_name}.")
                    break

                # Place the object at a designated location (for demonstration, use same position)
                if 'place' in globals():
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=50,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=5.0
                    )
                    if done:
                        print(f"[Task] Task ended after placing {obj_name}.")
                        break

            except Exception as e:
                print(f"[Task] Exception during pick/place of {obj_name}: {e}")
                continue

        print("===== Task Execution Complete =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
