# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'gripper1': ..., 'drawer1': ..., 'object_1': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (rotated gripper1 ninety_deg)
        # The missing predicate is likely (rotated gripper1 ninety_deg)
        # We need to ensure the gripper is rotated to ninety_deg before certain actions (e.g., move-to-side)
        # We'll perform an exploration step to check/set this state.

        # For demonstration, let's assume the gripper is named 'gripper1' and the angle is 'ninety_deg'
        gripper_name = 'gripper1'
        angle_ninety = 'ninety_deg'

        # Try to get the gripper's current rotation state from the environment, if possible
        # If not available, we will just perform the rotate action to ensure the predicate holds

        # The rotate skill is available: rotate(env, task, gripper, from_angle, to_angle)
        # We'll need to know the current angle. If not available, we can try both or just rotate to ninety_deg.

        # For this example, let's assume the initial angle is 'zero_deg'
        initial_angle = 'zero_deg'

        print(f"[Exploration] Rotating {gripper_name} from {initial_angle} to {angle_ninety} to ensure (rotated {gripper_name} {angle_ninety}) holds.")
        try:
            obs, reward, done = rotate(env, task, gripper_name, initial_angle, angle_ninety)
            if done:
                print("[Exploration] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate: {e}")

        # === Main Task Plan ===
        # Here you would continue with the oracle plan, using only the available skills.
        # For demonstration, let's assume the next step is to move the gripper to a target position.

        # Example: Move gripper1 to a specific position (e.g., to the side of a drawer)
        # We'll use the 'move' skill: move(env, task, target_pos, ...)
        # Let's assume the target position is given by positions['drawer1_side']

        if 'drawer1_side' in positions:
            target_pos = positions['drawer1_side']
            print(f"[Task] Moving {gripper_name} to drawer1_side at {target_pos}")
            try:
                obs, reward, done = move(env, task, target_pos)
                if done:
                    print("[Task] Task ended during move!")
                    return
            except Exception as e:
                print(f"[Task] Exception during move: {e}")
        else:
            print("[Task] drawer1_side position not found in positions dictionary.")

        # Example: Pull the drawer after moving to the correct position and orientation
        # We'll use the 'pull' skill if available
        # Let's assume the drawer is named 'drawer1'
        drawer_name = 'drawer1'
        try:
            print(f"[Task] Pulling {drawer_name} with {gripper_name}")
            obs, reward, done = pull(env, task, gripper_name, drawer_name)
            if done:
                print("[Task] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")

        # Continue with further steps as required by the oracle plan, using only the available skills:
        # e.g., pick, place, etc.
        # Example: pick an object after opening the drawer
        if 'object_1' in positions and 'drawer1_inside' in positions:
            object_name = 'object_1'
            object_pos = positions['object_1']
            print(f"[Task] Picking {object_name} at {object_pos}")
            try:
                obs, reward, done = pick(env, task, object_name, 'drawer1_inside')
                if done:
                    print("[Task] Task ended during pick!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pick: {e}")

        # Example: place the object at a target location
        if 'target_location' in positions:
            target_location = positions['target_location']
            print(f"[Task] Placing object at {target_location}")
            try:
                obs, reward, done = place(env, task, object_name, 'target_location')
                if done:
                    print("[Task] Task ended during place!")
                    return
            except Exception as e:
                print(f"[Task] Exception during place: {e}")

        print("[Task] Task plan execution completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()