# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def is_within_reach(gripper_pos, target_pos, max_distance=0.20):
    """Check if the target object is within reach of the gripper."""
    dist = np.linalg.norm(np.array(gripper_pos) - np.array(target_pos))
    print(f"[Check] Gripper at {gripper_pos}, target at {target_pos}, distance: {dist:.3f}")
    return dist <= max_distance

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()
        print("[Info] Object positions:", positions)

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that before picking, we must ensure the object is identified (i.e., present and within reach).
        # We'll perform a reachability check and presence check before pick.

        # For demonstration, let's assume we want to pick up 'object_1' and place it at 'target_location'
        # Replace these keys with actual object/location names from your environment
        object_to_pick = None
        target_location = None

        # Try to find a pickable object and a target location
        for key in positions:
            if "object" in key or "obj" in key:
                object_to_pick = key
                break
        for key in positions:
            if "target" in key or "goal" in key or "bin" in key or "place" in key or "location" in key:
                target_location = key
                break

        if object_to_pick is None or target_location is None:
            print("[Error] Could not find suitable object or target location in positions dictionary.")
            return

        object_pos = positions[object_to_pick]
        place_pos = positions[target_location]

        print(f"[Task] Will pick '{object_to_pick}' at {object_pos} and place at '{target_location}' at {place_pos}")

        # === Step 1: Exploration/Identification (simulate 'identified' predicate) ===
        # In a real plan, this might be a move or sensing action. Here, we check presence and reachability.
        obs = task.get_observation()
        gripper_pos = obs.gripper_pose[:3]
        if not is_within_reach(gripper_pos, object_pos, max_distance=0.25):
            print("[Exploration] Object not within reach. Attempting to move closer (if move skill available).")
            # If 'move' skill is available, use it to approach the object
            if 'move' in ['pick', 'place', 'move', 'rotate', 'pull']:
                try:
                    # For demonstration, we assume move(env, task, target_pos) signature
                    # If your move skill requires different parameters, adjust accordingly
                    print(f"[Action] Moving gripper closer to '{object_to_pick}' at {object_pos}")
                    obs, reward, done = move(env, task, target_pos=object_pos)
                    if done:
                        print("[Task] Task ended during move!")
                        return
                    obs = task.get_observation()
                    gripper_pos = obs.gripper_pose[:3]
                except Exception as e:
                    print(f"[Error] Move skill failed: {e}")
                    return
            else:
                print("[Error] No move skill available to approach object.")
                return

        # After move, check again
        if not is_within_reach(gripper_pos, object_pos, max_distance=0.20):
            print("[Error] Object still not within reach after move. Aborting pick.")
            return

        # === Step 2: Pick the Object (with safety and presence check) ===
        try:
            print(f"[Action] Attempting to pick '{object_to_pick}' at {object_pos}")
            obs, reward, done = pick(
                env,
                task,
                target_pos=object_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Error] Pick skill failed: {e}")
            return

        # === Step 3: Place the Object ===
        try:
            print(f"[Action] Attempting to place '{object_to_pick}' at '{target_location}' ({place_pos})")
            obs, reward, done = place(
                env,
                task,
                target_pos=place_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after place!")
                return
        except Exception as e:
            print(f"[Error] Place skill failed: {e}")
            return

        print("[Task] Pick and place completed successfully.")

        # === (Optional) Further actions: rotate, pull, etc. ===
        # If your plan requires, you can add rotate/pull actions here using the same pattern.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()