# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (is-open bottom) ===
        # Feedback indicates that the predicate (is-open bottom) is missing.
        # We need to explore the environment to determine if the 'bottom' drawer is open.
        # The only available skills are: ['pick', 'place', 'move', 'rotate', 'pull']
        # We will use 'move' to approach the drawer, and 'pull' to attempt to open it.
        # If the drawer is already open, 'pull' should have no effect or fail gracefully.

        # --- Identify the 'bottom' drawer and its position ---
        # We assume 'bottom' is the name of the drawer object.
        # The positions dictionary should contain the position of the 'bottom' drawer.
        if 'bottom' not in positions:
            print("[Exploration] Error: 'bottom' drawer not found in object positions.")
            return

        bottom_drawer_pos = positions['bottom']
        print(f"[Exploration] Found 'bottom' drawer at position: {bottom_drawer_pos}")

        # --- Move the gripper to the front of the 'bottom' drawer ---
        # We assume the gripper should approach the drawer from the front (along -y or +y axis).
        # For simplicity, we use the drawer's position directly.
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(bottom_drawer_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during move to bottom drawer!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")
            return

        # --- Attempt to pull the drawer to check if it is open ---
        # The 'pull' skill may require additional parameters, but we use only the available interface.
        # If the drawer is already open, this should have no effect.
        try:
            obs, reward, done = pull(
                env,
                task,
                drawer_name='bottom'
            )
            if done:
                print("[Exploration] Task ended during pull action!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")
            # If pull fails, it may indicate the drawer is already open or not gripped.

        # --- Check if the drawer is open (predicate is-open bottom) ---
        # We attempt to infer the state by checking the drawer's position or state in the observation.
        # This is a placeholder for actual predicate checking logic.
        try:
            obs = task.get_observation()
            # Assume the observation contains a dictionary or attribute for drawer states
            is_open = False
            if hasattr(obs, 'drawer_states'):
                # drawer_states: dict of {drawer_name: is_open}
                is_open = obs.drawer_states.get('bottom', False)
            elif hasattr(obs, 'is_open_bottom'):
                is_open = getattr(obs, 'is_open_bottom')
            else:
                # Fallback: check position change (e.g., y-axis displacement)
                current_pos = positions.get('bottom', None)
                if current_pos is not None:
                    # If the drawer's y position is significantly displaced, consider it open
                    if abs(current_pos[1] - bottom_drawer_pos[1]) > 0.05:
                        is_open = True
            print(f"[Exploration] Drawer 'bottom' open state: {is_open}")
            if is_open:
                print("[Exploration] Predicate (is-open bottom) is TRUE.")
            else:
                print("[Exploration] Predicate (is-open bottom) is FALSE.")
        except Exception as e:
            print(f"[Exploration] Exception during open state check: {e}")

        # === End of Exploration Phase ===

        # === (Optional) Continue with Oracle Plan or Task-Specific Actions ===
        # Here you would continue with the main plan, e.g., pick/place actions, etc.
        # For this exploration, we focus on the missing predicate as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()