# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (is-side-pos plate_approach_pos ?d)
        # The missing predicate is (is-side-pos plate_approach_pos ?d)
        # We need to explore and check if 'plate_approach_pos' is a side position for the drawer (?d)
        # Since we do not have a skill to directly check predicates, we will attempt to move the gripper to 'plate_approach_pos'
        # and see if the move is successful (i.e., no collision or error), which would indicate that the position is valid.

        # For demonstration, let's assume the drawer is named 'drawer' and the side position is 'plate_approach_pos'
        # These names should match the actual object names in your environment.
        drawer_name = None
        plate_approach_pos_name = None

        # Try to find the drawer and plate_approach_pos from positions
        for key in positions.keys():
            if 'drawer' in key and drawer_name is None:
                drawer_name = key
            if 'plate_approach_pos' in key and plate_approach_pos_name is None:
                plate_approach_pos_name = key

        if drawer_name is None or plate_approach_pos_name is None:
            print("[Exploration] Could not find required objects in positions dictionary.")
            print("Available keys:", list(positions.keys()))
            return

        plate_approach_pos = np.array(positions[plate_approach_pos_name])

        print(f"[Exploration] Attempting to move to side position '{plate_approach_pos_name}' for drawer '{drawer_name}'.")

        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=plate_approach_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Successfully moved to '{plate_approach_pos_name}'. Predicate (is-side-pos {plate_approach_pos_name} {drawer_name}) is likely TRUE.")
        except Exception as e:
            print(f"[Exploration] Failed to move to '{plate_approach_pos_name}'. Exception: {e}")
            print(f"[Exploration] Predicate (is-side-pos {plate_approach_pos_name} {drawer_name}) is likely FALSE or not defined.")
            return

        # === Main Task Plan (Example) ===
        # After exploration, proceed with the oracle plan using only available skills.
        # For demonstration, let's assume the plan is:
        # 1. Move to plate_approach_pos (already done in exploration)
        # 2. Rotate gripper if needed (e.g., to 'ninety_deg')
        # 3. Move to anchor position (if required)
        # 4. Pick the drawer handle (if required)
        # 5. Pull the drawer open

        # Step 2: Rotate gripper to 'ninety_deg' (if such an angle exists)
        # We'll try to find an angle named 'ninety_deg' in positions or use a default
        angle_name = None
        for key in positions.keys():
            if 'ninety_deg' in key:
                angle_name = key
                break
        # If not found, skip rotation (or use a default value if your environment requires it)
        if angle_name is not None:
            print(f"[Task] Rotating gripper to angle '{angle_name}'.")
            try:
                obs, reward, done = rotate(
                    env,
                    task,
                    gripper_name='gripper',  # Replace with actual gripper name if needed
                    from_angle='zero_deg',   # Replace with actual current angle if known
                    to_angle=angle_name
                )
            except Exception as e:
                print(f"[Task] Exception during rotate: {e}")
        else:
            print("[Task] No 'ninety_deg' angle found; skipping rotate.")

        # Step 3: Move to anchor position (if such a position exists)
        anchor_pos_name = None
        for key in positions.keys():
            if 'anchor' in key:
                anchor_pos_name = key
                break
        if anchor_pos_name is not None:
            anchor_pos = np.array(positions[anchor_pos_name])
            print(f"[Task] Moving to anchor position '{anchor_pos_name}'.")
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=anchor_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Task] Exception during move to anchor: {e}")
        else:
            print("[Task] No anchor position found; skipping move to anchor.")

        # Step 4: Pick the drawer handle (if such a skill is available)
        # Since 'pick' is available, try to pick the drawer handle if its position is known
        drawer_handle_name = None
        for key in positions.keys():
            if 'handle' in key and drawer_name in key:
                drawer_handle_name = key
                break
        if drawer_handle_name is not None:
            drawer_handle_pos = np.array(positions[drawer_handle_name])
            print(f"[Task] Picking drawer handle '{drawer_handle_name}'.")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=drawer_handle_pos,
                    approach_distance=0.10,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Task] Exception during pick: {e}")
        else:
            print("[Task] No drawer handle found; skipping pick.")

        # Step 5: Pull the drawer open
        print(f"[Task] Attempting to pull the drawer '{drawer_name}'.")
        try:
            obs, reward, done = pull(
                env,
                task,
                drawer_name=drawer_name
            )
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")

        # Task complete
        print("[Task] Oracle plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()