# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Robust Object Existence Check ===
        # Instead of hard-coding object names, check which objects are present
        # For exploration, we want to find out if any required predicate is missing
        # and which objects are available for manipulation.

        # List of candidate objects to manipulate (e.g., tomatoes, plate, etc.)
        candidate_objects = []
        for obj_name in positions.keys():
            # Filter out non-manipulable objects if needed (e.g., background, table, etc.)
            if "tomato" in obj_name or "plate" in obj_name or "drawer" in obj_name:
                candidate_objects.append(obj_name)

        print(f"[Exploration] Candidate objects found: {candidate_objects}")

        # If no tomatoes found, warn and exit
        tomatoes = [obj for obj in candidate_objects if "tomato" in obj]
        if not tomatoes:
            print("[Exploration] No tomato objects found in the environment! Please check object list.")
            return

        # Find plate position if available
        plate_candidates = [obj for obj in candidate_objects if "plate" in obj]
        if plate_candidates:
            plate_name = plate_candidates[0]
            plate_pos = positions[plate_name]
            print(f"[Exploration] Plate found: {plate_name} at {plate_pos}")
        else:
            print("[Exploration] No plate found in the environment! Please check object list.")
            return

        # === Exploration Phase: Check for Missing Predicates ===
        # Try to pick and place each tomato onto the plate, catching errors to identify missing predicates
        for tomato in tomatoes:
            print(f"[Exploration] Attempting to pick and place {tomato} onto {plate_name}.")

            # Get tomato position
            tomato_pos = positions.get(tomato, None)
            if tomato_pos is None:
                print(f"[Exploration] Position for {tomato} not found, skipping.")
                continue

            # Safety: Check if gripper is empty before pick
            obs = task.get_observation()
            if hasattr(obs, 'gripper_open') and obs.gripper_open < 0.5:
                print("[Safety] Gripper not open, attempting to open before pick.")
                # Try to open gripper (simulate handempty)
                try:
                    # Use place with current position to open gripper
                    current_pos = obs.gripper_pose[:3]
                    place(env, task, current_pos)
                except Exception as e:
                    print(f"[Safety] Failed to open gripper: {e}")

            # Try to pick the tomato
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=tomato_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Picked {tomato}.")
            except Exception as e:
                print(f"[Exploration] Failed to pick {tomato}: {e}")
                print("[Exploration] This may indicate a missing predicate or precondition (e.g., handempty, at, identified, etc.).")
                continue

            if done:
                print("[Exploration] Task ended after pick! Exiting.")
                return

            # Try to place the tomato on the plate
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=plate_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Placed {tomato} on {plate_name}.")
            except Exception as e:
                print(f"[Exploration] Failed to place {tomato}: {e}")
                print("[Exploration] This may indicate a missing predicate or precondition (e.g., holding, plate position, etc.).")
                continue

            if done:
                print("[Exploration] Task ended after place! Exiting.")
                return

        # === Exploration for Drawer/Other Skills (if present) ===
        # If there is a drawer, try to open it using rotate, move, pick-drawer, pull, etc.
        drawer_candidates = [obj for obj in candidate_objects if "drawer" in obj]
        if drawer_candidates:
            drawer_name = drawer_candidates[0]
            print(f"[Exploration] Drawer found: {drawer_name}. Attempting to open.")

            # For demonstration, try to rotate, move-to-side, move-to-anchor, pick-drawer, pull
            # These require more detailed position/angle info, so check if available
            # For now, just attempt rotate and pull if possible
            try:
                # Example: rotate gripper to ninety_deg (if such angle exists)
                obs = task.get_observation()
                if hasattr(obs, 'gripper_angle'):
                    from_angle = obs.gripper_angle
                    to_angle = np.pi/2  # 90 degrees in radians
                    rotate(env, task, from_angle, to_angle)
                    print(f"[Exploration] Rotated gripper from {from_angle} to {to_angle}.")
            except Exception as e:
                print(f"[Exploration] Failed to rotate gripper: {e}")

            try:
                # Try to pull the drawer (simulate pull action)
                pull(env, task, drawer_name)
                print(f"[Exploration] Pulled {drawer_name}.")
            except Exception as e:
                print(f"[Exploration] Failed to pull {drawer_name}: {e}")
                print("[Exploration] This may indicate a missing predicate or precondition (e.g., holding-drawer, is-locked, etc.).")

        print("[Exploration] Exploration phase complete. If any actions failed, check logs for missing predicates or preconditions.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
