# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The exploration domain suggests that after certain actions, predicates like
        # identified, temperature-known, weight-known, durability-known, lock-known
        # become true. We need to explore which predicate is missing in the main domain.

        # We'll use the available skills: ['pick', 'place', 'move', 'rotate', 'pull']
        # and try to observe the effect of each on the environment.

        # For demonstration, let's try to pick an object and see if any predicate is missing.
        # We'll also try to move and pull, as these are related to the exploration predicates.

        # We'll use try/except to handle missing objects or type mismatches gracefully.

        # 1. Try to pick an object (if any available)
        picked = False
        for obj_name, obj_pos in positions.items():
            try:
                print(f"[Exploration] Attempting to pick object: {obj_name} at {obj_pos}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                picked = True
                print(f"[Exploration] Successfully picked {obj_name}")
                if done:
                    print("[Exploration] Task ended after pick!")
                    return
                break
            except Exception as e:
                print(f"[Exploration] Could not pick {obj_name}: {e}")
                continue

        # 2. If pick succeeded, try to place the object at a new location (if possible)
        if picked:
            # Find a different location to place
            for loc_name, loc_pos in positions.items():
                if loc_name != obj_name:
                    try:
                        print(f"[Exploration] Attempting to place object: {obj_name} at {loc_pos}")
                        obs, reward, done = place(
                            env,
                            task,
                            target_pos=loc_pos,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        print(f"[Exploration] Successfully placed {obj_name} at {loc_name}")
                        if done:
                            print("[Exploration] Task ended after place!")
                            return
                        break
                    except Exception as e:
                        print(f"[Exploration] Could not place {obj_name} at {loc_name}: {e}")
                        continue

        # 3. Try to rotate the gripper (if skill is available and relevant)
        try:
            # For demonstration, rotate to a fixed quaternion (identity)
            target_quat = np.array([0, 0, 0, 1])
            print("[Exploration] Attempting to rotate gripper to identity quaternion")
            obs, reward, done = rotate(
                env,
                task,
                target_quat=target_quat,
                max_steps=50,
                threshold=0.05,
                timeout=5.0
            )
            print("[Exploration] Successfully rotated gripper")
            if done:
                print("[Exploration] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Could not rotate gripper: {e}")

        # 4. Try to pull (if any object is suitable)
        for obj_name, obj_pos in positions.items():
            try:
                print(f"[Exploration] Attempting to pull object: {obj_name} at {obj_pos}")
                obs, reward, done = pull(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Successfully pulled {obj_name}")
                if done:
                    print("[Exploration] Task ended after pull!")
                    return
                break
            except Exception as e:
                print(f"[Exploration] Could not pull {obj_name}: {e}")
                continue

        # 5. Try to move (if skill is available and relevant)
        # Since 'move' is in available skills, but not implemented in skill_code, we skip actual call.

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
