# run_skeleton_task.py (Completed with Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate ===
        # The feedback and exploration domain suggest that some predicates (e.g., lock-known, durability-known, weight-known, temperature-known, identified) may be missing.
        # We will attempt to use the available skills to probe the environment and infer which predicate is missing.

        # 1. Try to move to each location and check for identification/temperature-known predicates
        # 2. Try to pick objects and check for weight-known/durability-known predicates
        # 3. Try to pull drawers/objects and check for lock-known predicate

        # For demonstration, we will iterate over all objects and locations in the environment.
        # The actual object and location names should be obtained from the observation or positions.

        # Example: positions = {'drawer1': (x1, y1, z1), 'object1': (x2, y2, z2), ...}
        # We'll use the keys as object names.

        # For this exploration, we assume the following conventions:
        # - Any key containing 'drawer' is a drawer object
        # - Any key containing 'object' or 'item' is a manipulable object
        # - Any key containing 'loc' or 'position' is a location

        # Collect object and location names
        object_names = []
        drawer_names = []
        location_names = []
        for name in positions.keys():
            lname = name.lower()
            if 'drawer' in lname:
                drawer_names.append(name)
            elif 'object' in lname or 'item' in lname:
                object_names.append(name)
            elif 'loc' in lname or 'position' in lname or 'place' in lname:
                location_names.append(name)
        # Fallback: if no explicit locations, use all unique positions
        if not location_names:
            location_names = list(set(positions.keys()) - set(drawer_names) - set(object_names))

        # If no objects found, just use all keys as objects (for robustness)
        if not object_names:
            object_names = list(set(positions.keys()) - set(drawer_names) - set(location_names))

        # === 1. Exploration: Try to pick each object and see what knowledge is gained ===
        for obj in object_names:
            print(f"[Exploration] Attempting to pick object: {obj}")
            # Find a plausible location for the object
            obj_pos = positions[obj]
            # For this exploration, we assume the robot is already at the correct location
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Picked {obj}. Checking for new knowledge (weight-known, durability-known, etc.)")
                # Here, in a real system, we would check the state for new predicates
                # For this skeleton, we just log the action
                if done:
                    print(f"[Exploration] Task ended after picking {obj}!")
                    return
            except Exception as e:
                print(f"[Exploration] Could not pick {obj}: {e}")

        # === 2. Exploration: Try to pull each drawer and see if lock-known is discovered ===
        for drawer in drawer_names:
            print(f"[Exploration] Attempting to pull drawer: {drawer}")
            drawer_pos = positions[drawer]
            try:
                # For pulling, we may need to first pick the drawer handle (simulate as pick), then pull
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=drawer_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking drawer {drawer}!")
                    return
                obs, reward, done = pull(
                    env,
                    task,
                    target_pos=drawer_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Pulled {drawer}. Checking for lock-known predicate.")
                if done:
                    print(f"[Exploration] Task ended after pulling {drawer}!")
                    return
            except Exception as e:
                print(f"[Exploration] Could not pull {drawer}: {e}")

        # === 3. Exploration: Try to rotate each object (if applicable) and check for stability/force calibration ===
        for obj in object_names + drawer_names:
            print(f"[Exploration] Attempting to rotate object: {obj}")
            obj_pos = positions[obj]
            # For rotation, we need a target quaternion; for exploration, use a fixed rotation (e.g., 90 degrees about z)
            try:
                # Check if object is present in the environment (feedback #1)
                if obj not in positions:
                    print(f"[rotate] Target object '{obj}' not found in object list.")
                    continue
                # Safety check: (feedback #3) - in real system, check stability
                # For this skeleton, we just log the check
                print(f"[rotate] Checking stability of '{obj}' (simulated check).")
                # Force calibration (feedback #2) - in real system, would measure force
                print(f"[rotate] Performing force calibration for '{obj}' (simulated).")
                # Attempt rotation
                # For demonstration, use a quaternion representing 90 deg about z
                from scipy.spatial.transform import Rotation as R
                quat_90z = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
                # RLBench expects wxyz, so reorder
                quat_90z = np.roll(quat_90z, 1)
                obs, reward, done = rotate(
                    env,
                    task,
                    target_quat=quat_90z,
                    max_steps=100,
                    threshold=0.05,
                    timeout=10.0
                )
                print(f"[Exploration] Rotated {obj}. Checking for rotation success/failure.")
                if done:
                    print(f"[Exploration] Task ended after rotating {obj}!")
                    return
            except Exception as e:
                print(f"[Exploration] Could not rotate {obj}: {e}")

        # === 4. Exploration: Try to place each object at each location ===
        for obj in object_names:
            for loc in location_names:
                print(f"[Exploration] Attempting to place object {obj} at location {loc}")
                loc_pos = positions[loc]
                try:
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=loc_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Exploration] Placed {obj} at {loc}.")
                    if done:
                        print(f"[Exploration] Task ended after placing {obj} at {loc}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Could not place {obj} at {loc}: {e}")

        # === Exploration Summary ===
        print("[Exploration] Completed all exploration actions. Review logs to determine which predicate is missing based on observed effects and feedback.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()