# run_skeleton_task.py (Completed Executable with Exploration Phase and Feedback Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

import traceback

def euler_from_quat(quat):
    # Converts quaternion (x, y, z, w) to Euler angles (roll, pitch, yaw)
    # Assumes input is np.array or list of 4 elements
    x, y, z, w = quat
    t0 = +2.0 * (w * x + y * z)
    t1 = +1.0 - 2.0 * (x * x + y * y)
    roll_x = np.arctan2(t0, t1)
    t2 = +2.0 * (w * y - z * x)
    t2 = np.clip(t2, -1.0, 1.0)
    pitch_y = np.arcsin(t2)
    t3 = +2.0 * (w * z + x * y)
    t4 = +1.0 - 2.0 * (y * y + z * z)
    yaw_z = np.arctan2(t3, t4)
    return np.array([roll_x, pitch_y, yaw_z])

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration and feedback handling.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The goal is to explore the environment to determine which predicate is missing or unknown.
        # We will use the available skills to interact with objects and locations, and log the results.
        # This phase is based on the exploration domain knowledge.

        print("===== [Exploration Phase] Start =====")
        # Example: Try to move to each location and pick/place objects to see which predicates become known.
        # This is a generic exploration loop; adapt as needed for your environment.

        # Get all object and location names from positions
        object_names = [name for name in positions if not name.startswith('location_')]
        location_names = [name for name in positions if name.startswith('location_')]

        # If no explicit location names, try to infer from object positions
        if not location_names:
            # Assume all unique position tuples are locations
            unique_locs = set(tuple(pos) for pos in positions.values())
            location_names = [f"loc_{i}" for i, _ in enumerate(unique_locs)]

        # Try to move to each location (if move skill is available)
        if 'move' in ['pick', 'place', 'move', 'rotate', 'pull']:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to move to {loc}")
                    # The move skill may require from/to; here we use a dummy 'from' (could be current location)
                    # If the move skill requires more info, adapt accordingly.
                    # Since we don't have robot location, we use the first location as 'from'
                    from_loc = location_names[0]
                    if loc != from_loc:
                        obs, reward, done = move(env, task, from_loc, loc)
                        print(f"[Exploration] Moved from {from_loc} to {loc}.")
                        if done:
                            print("[Exploration] Task ended during move!")
                            break
                except Exception as e:
                    print(f"[Exploration] Move to {loc} failed: {e}")
                    traceback.print_exc()

        # Try to pick each object at its location (if pick skill is available)
        if 'pick' in ['pick', 'place', 'move', 'rotate', 'pull']:
            for obj in object_names:
                try:
                    obj_pos = positions[obj]
                    # Find the closest location to the object
                    closest_loc = None
                    min_dist = float('inf')
                    for loc in location_names:
                        loc_pos = positions.get(loc, None)
                        if loc_pos is not None:
                            dist = np.linalg.norm(np.array(obj_pos) - np.array(loc_pos))
                            if dist < min_dist:
                                min_dist = dist
                                closest_loc = loc
                    if closest_loc is not None:
                        print(f"[Exploration] Attempting to pick {obj} at {closest_loc}")
                        obs, reward, done = pick(env, task, obj, closest_loc)
                        print(f"[Exploration] Picked {obj} at {closest_loc}.")
                        if done:
                            print("[Exploration] Task ended during pick!")
                            break
                except Exception as e:
                    print(f"[Exploration] Pick {obj} failed: {e}")
                    traceback.print_exc()

        # Try to place each object at each location (if place skill is available)
        if 'place' in ['pick', 'place', 'move', 'rotate', 'pull']:
            for obj in object_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting to place {obj} at {loc}")
                        obs, reward, done = place(env, task, obj, loc)
                        print(f"[Exploration] Placed {obj} at {loc}.")
                        if done:
                            print("[Exploration] Task ended during place!")
                            break
                    except Exception as e:
                        print(f"[Exploration] Place {obj} at {loc} failed: {e}")
                        traceback.print_exc()

        # Try to rotate the gripper to various angles (if rotate skill is available)
        if 'rotate' in ['pick', 'place', 'move', 'rotate', 'pull']:
            try:
                # Example: rotate to 0, 90, 180 degrees (in radians)
                angles = [0, np.pi/2, np.pi]
                for angle in angles:
                    # Convert angle to quaternion (about z-axis)
                    q = np.array([0, 0, np.sin(angle/2), np.cos(angle/2)])
                    print(f"[Exploration] Attempting to rotate gripper to angle {angle} rad")
                    obs, reward, done = rotate(env, task, q)
                    print(f"[Exploration] Rotated gripper to angle {angle} rad.")
                    if done:
                        print("[Exploration] Task ended during rotate!")
                        break
            except Exception as e:
                print(f"[Exploration] Rotate failed: {e}")
                traceback.print_exc()

        # Try to pull each object (if pull skill is available)
        if 'pull' in ['pick', 'place', 'move', 'rotate', 'pull']:
            for obj in object_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting to pull {obj} at {loc}")
                        obs, reward, done = pull(env, task, obj, loc)
                        print(f"[Exploration] Pulled {obj} at {loc}.")
                        if done:
                            print("[Exploration] Task ended during pull!")
                            break
                    except Exception as e:
                        print(f"[Exploration] Pull {obj} at {loc} failed: {e}")
                        traceback.print_exc()

        print("===== [Exploration Phase] End =====")

        # === Main Task Plan Execution ===
        # After exploration, proceed with the oracle plan (if known).
        # For this generic skeleton, you would insert the actual plan steps here using the available skills.
        # Example:
        # obs, reward, done = pick(env, task, 'object_1', 'location_1')
        # if done:
        #     print("[Task] Task ended after pick!")
        #     return

        # TODO: Insert actual oracle plan steps here as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()