# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate ===
        # Feedback: (rotated gripper ninety_deg)
        # The plan requires the predicate (rotated gripper ninety_deg) to be true before certain actions.
        # We must ensure the gripper is rotated to ninety_deg before proceeding.

        # For demonstration, let's assume the following:
        # - There is a 'gripper' object whose pose we can get from positions['gripper']
        # - The 'rotate' skill is available and takes (env, task, gripper_name, from_angle, to_angle)
        # - The initial angle is not ninety_deg, so we must rotate it.

        # The actual object/angle names may differ; adjust as needed for your environment.
        # We'll use the available skills: ['pick', 'place', 'move', 'rotate', 'pull']

        # --- Example Plan Execution ---

        # 1. Ensure gripper is rotated to ninety_deg
        gripper_name = 'gripper'
        from_angle = 'zero_deg'
        to_angle = 'ninety_deg'

        # If the environment provides the current angle, check it; otherwise, always rotate.
        try:
            print("[Exploration] Rotating gripper to ninety_deg if needed...")
            obs, reward, done = rotate(env, task, gripper_name, from_angle, to_angle)
            if done:
                print("[Exploration] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate: {e}")

        # 2. Example: Pick an object after rotating gripper
        # Let's assume we want to pick 'object_1' at its position
        object_to_pick = 'object_1'
        if object_to_pick in positions:
            target_pos = np.array(positions[object_to_pick])
            try:
                print(f"[Task] Picking {object_to_pick} at {target_pos} ...")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=target_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after pick!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pick: {e}")
        else:
            print(f"[Task] {object_to_pick} not found in positions.")

        # 3. Example: Place the object at a target location (if defined)
        place_location = 'place_zone'
        if place_location in positions:
            place_pos = np.array(positions[place_location])
            try:
                print(f"[Task] Placing {object_to_pick} at {place_pos} ...")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=place_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Task] Exception during place: {e}")
        else:
            print(f"[Task] {place_location} not found in positions.")

        # 4. (Optional) Pull a drawer if needed
        drawer_name = 'drawer'
        if drawer_name in positions:
            try:
                print(f"[Task] Pulling {drawer_name} ...")
                obs, reward, done = pull(env, task, drawer_name)
                if done:
                    print("[Task] Task ended after pull!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pull: {e}")
        else:
            print(f"[Task] {drawer_name} not found in positions.")

        # 5. (Optional) Move to another location if needed
        # If 'move' skill is available and required, use it as follows:
        # move(env, task, from_location, to_location)
        # (Not used here as no specific move is required by feedback.)

        print("[Task] Plan execution completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
