# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x,y,z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (holding gripper1)
        # The feedback suggests that after a pick action, the predicate (holding gripper1) is missing.
        # We need to explore which predicate is missing after pick, using only available skills.

        # For exploration, we will attempt to pick an object and check if the state reflects the expected predicate.
        # Since we cannot directly manipulate predicates, we use the available skills and observe the result.

        # Find a pickable object and its position
        pickable_obj = None
        pickable_obj_pos = None
        for obj_name, pos in positions.items():
            if "gripper" not in obj_name.lower() and "drawer" not in obj_name.lower():
                pickable_obj = obj_name
                pickable_obj_pos = pos
                break
        if pickable_obj is None:
            # Fallback: just pick the first object
            pickable_obj = list(positions.keys())[0]
            pickable_obj_pos = positions[pickable_obj]

        print(f"[Exploration] Attempting to pick object: {pickable_obj} at {pickable_obj_pos}")

        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(pickable_obj_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")
            shutdown_environment(env)
            return

        # After pick, check if the robot is holding the object (predicate check)
        # Since we cannot access predicates directly, we infer from the environment/observation
        obs = task.get_observation()
        # For demonstration, print out gripper state or held object info if available
        if hasattr(obs, 'holding') and obs.holding:
            print(f"[Exploration] Robot is holding: {obs.holding}")
        else:
            print("[Exploration] Unable to verify holding predicate directly. Possible missing predicate: (holding gripper1)")

        # === End of Exploration Phase ===

        # The missing predicate is likely (holding gripper1) after pick.
        # In a real plan, you would ensure that after pick, the state reflects the robot is holding the object.

        # === Continue with Task Plan (if any) ===
        # For demonstration, place the object back if possible
        place_pos = None
        for obj_name, pos in positions.items():
            if obj_name != pickable_obj:
                place_pos = pos
                break
        if place_pos is not None:
            print(f"[Task] Placing object {pickable_obj} at {place_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(place_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Task] Exception during place: {e}")
                shutdown_environment(env)
                return

        print("[Task] Exploration and demonstration complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()