# run_skeleton_task.py (Completed Executable Code for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try to Discover Missing Predicate ===
        # Based on feedback, the planner times out, likely due to a missing predicate in the domain.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use available skills to interact with objects and observe if any information is missing or not updated.

        # For demonstration, we will:
        # 1. Move to each object location (if possible)
        # 2. Attempt to pick the object (if possible)
        # 3. Attempt to place the object at a new location (if possible)
        # 4. Attempt to rotate or pull if the object is a drawer or similar

        # This will help us determine if any predicate (e.g., lock-known, identified, etc.) is missing from the domain.

        # Get all object names and their positions
        object_names = list(positions.keys())
        print("[Exploration] Objects found in environment:", object_names)

        # For each object, try to interact using available skills
        for obj_name in object_names:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Attempting to move to object '{obj_name}' at position {obj_pos}")

            # 1. Move to the object's position
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Moved to '{obj_name}' successfully.")
                if done:
                    print("[Exploration] Task ended during move!")
                    return
            except Exception as e:
                print(f"[Exploration] Failed to move to '{obj_name}': {e}")
                continue

            # 2. Try to pick the object
            try:
                print(f"[Exploration] Attempting to pick '{obj_name}'")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Picked '{obj_name}' successfully.")
                if done:
                    print("[Exploration] Task ended during pick!")
                    return
            except Exception as e:
                print(f"[Exploration] Failed to pick '{obj_name}': {e}")

            # 3. Try to place the object at a slightly offset position (if pick succeeded)
            try:
                place_offset = np.array([0.05, 0, 0])  # Offset in x-direction
                place_pos = np.array(obj_pos) + place_offset
                print(f"[Exploration] Attempting to place '{obj_name}' at {place_pos}")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=place_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Placed '{obj_name}' successfully.")
                if done:
                    print("[Exploration] Task ended during place!")
                    return
            except Exception as e:
                print(f"[Exploration] Failed to place '{obj_name}': {e}")

            # 4. Try to rotate or pull if the object is a drawer (by name heuristic)
            if "drawer" in obj_name.lower():
                # Try rotate
                try:
                    print(f"[Exploration] Attempting to rotate '{obj_name}'")
                    # For demonstration, try rotating from 'zero_deg' to 'ninety_deg'
                    obs, reward, done = rotate(
                        env,
                        task,
                        gripper_name="gripper",  # Assumed gripper name
                        from_angle="zero_deg",
                        to_angle="ninety_deg"
                    )
                    print(f"[Exploration] Rotated '{obj_name}' successfully.")
                    if done:
                        print("[Exploration] Task ended during rotate!")
                        return
                except Exception as e:
                    print(f"[Exploration] Failed to rotate '{obj_name}': {e}")

                # Try pull
                try:
                    print(f"[Exploration] Attempting to pull '{obj_name}'")
                    obs, reward, done = pull(
                        env,
                        task,
                        gripper_name="gripper",  # Assumed gripper name
                        drawer_name=obj_name
                    )
                    print(f"[Exploration] Pulled '{obj_name}' successfully.")
                    if done:
                        print("[Exploration] Task ended during pull!")
                        return
                except Exception as e:
                    print(f"[Exploration] Failed to pull '{obj_name}': {e}")

        print("[Exploration] Completed exploration of all objects.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
