# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try All Available Skills to Detect Missing Predicate ===
        # The goal is to explore the environment using the available skills to trigger any missing predicate
        # (e.g., lock-known, identified, temperature-known, etc.) as suggested by the feedback and exploration knowledge.

        # List of available skills
        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']

        # Try to apply each skill to each object/position in the environment
        # This is a generic exploration: try to use each skill in a safe way to see if any predicate is missing
        print("=== [Exploration] Starting exploration phase to detect missing predicates ===")
        exploration_success = False
        for obj_name, obj_pos in positions.items():
            print(f"[Exploration] Trying skills on object: {obj_name} at {obj_pos}")
            # Try move: move gripper to the object's position
            if 'move' in available_skills:
                try:
                    print(f"  [Exploration] move to {obj_name}")
                    obs, reward, done = move(env, task, target_pos=np.array(obj_pos))
                    if done:
                        print(f"  [Exploration] Task ended after move to {obj_name}")
                        exploration_success = True
                        break
                except Exception as e:
                    print(f"  [Exploration] move failed for {obj_name}: {e}")

            # Try pick: attempt to pick the object
            if 'pick' in available_skills:
                try:
                    print(f"  [Exploration] pick {obj_name}")
                    # The pick skill may require additional parameters (e.g., approach distance, axis)
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"  [Exploration] Task ended after pick {obj_name}")
                        exploration_success = True
                        break
                except Exception as e:
                    print(f"  [Exploration] pick failed for {obj_name}: {e}")

            # Try place: attempt to place the object back at its position (if holding)
            if 'place' in available_skills:
                try:
                    print(f"  [Exploration] place {obj_name}")
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"  [Exploration] Task ended after place {obj_name}")
                        exploration_success = True
                        break
                except Exception as e:
                    print(f"  [Exploration] place failed for {obj_name}: {e}")

            # Try rotate: if the object is a gripper or has a rotatable part
            if 'rotate' in available_skills:
                try:
                    print(f"  [Exploration] rotate {obj_name}")
                    # For demonstration, try rotating to a fixed angle (e.g., 90 degrees)
                    # You may need to adapt this if you know the angles/rotatable objects
                    obs, reward, done = rotate(
                        env,
                        task,
                        gripper_name=obj_name,
                        from_angle='zero_deg',
                        to_angle='ninety_deg'
                    )
                    if done:
                        print(f"  [Exploration] Task ended after rotate {obj_name}")
                        exploration_success = True
                        break
                except Exception as e:
                    print(f"  [Exploration] rotate failed for {obj_name}: {e}")

            # Try pull: if the object is a drawer or can be pulled
            if 'pull' in available_skills:
                try:
                    print(f"  [Exploration] pull {obj_name}")
                    obs, reward, done = pull(
                        env,
                        task,
                        object_name=obj_name
                    )
                    if done:
                        print(f"  [Exploration] Task ended after pull {obj_name}")
                        exploration_success = True
                        break
                except Exception as e:
                    print(f"  [Exploration] pull failed for {obj_name}: {e}")

        if not exploration_success:
            print("[Exploration] Exploration phase completed. No terminal state reached. Check logs for missing predicate clues.")

        # === End of Exploration Phase ===

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you can proceed to execute the oracle plan if available.
        # For this template, we focus on the exploration phase as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
