# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer0': ..., 'gripper': ..., ...}
        positions = get_object_positions()

        # --- Exploration Phase: Check for missing predicate (is-open drawer0) ---
        # The feedback indicates that (is-open drawer0) is missing or needs to be explored.
        # We will attempt to open the drawer using the available skills and check the result.

        # Assumptions for object names (adjust as needed based on your environment):
        # - Drawer name: 'drawer0'
        # - Gripper name: 'gripper'
        # - Anchor position: 'anchor_pos_drawer0'
        # - Side position: 'side_pos_drawer0'
        # - Angles: 'zero_deg', 'ninety_deg'
        # These should be present in the positions dictionary or can be inferred.

        # Extract required positions and objects
        try:
            drawer_name = 'drawer0'
            gripper_name = 'gripper'
            anchor_pos_name = 'anchor_pos_drawer0'
            side_pos_name = 'side_pos_drawer0'
            zero_deg_name = 'zero_deg'
            ninety_deg_name = 'ninety_deg'

            drawer_pos = positions[drawer_name]
            gripper_pos = positions[gripper_name]
            anchor_pos = positions[anchor_pos_name]
            side_pos = positions[side_pos_name]
            zero_deg = positions[zero_deg_name]
            ninety_deg = positions[ninety_deg_name]
        except Exception as e:
            print("[Error] Required object or position not found in positions dictionary:", e)
            shutdown_environment(env)
            return

        # --- Step 1: Move gripper to side position (preparation) ---
        print("[Exploration] Moving gripper to side position near drawer.")
        try:
            obs, reward, done = move(env, task, target_pos=side_pos)
            if done:
                print("[Exploration] Task ended unexpectedly during move to side position.")
                return
        except Exception as e:
            print("[Error] Exception during move to side position:", e)
            return

        # --- Step 2: Rotate gripper to 90 degrees (if required) ---
        print("[Exploration] Rotating gripper to ninety_deg.")
        try:
            obs, reward, done = rotate(env, task, gripper_name, zero_deg, ninety_deg)
            if done:
                print("[Exploration] Task ended unexpectedly during rotate.")
                return
        except Exception as e:
            print("[Error] Exception during rotate:", e)
            return

        # --- Step 3: Move gripper to anchor position (for grasping drawer handle) ---
        print("[Exploration] Moving gripper to anchor position.")
        try:
            obs, reward, done = move(env, task, target_pos=anchor_pos)
            if done:
                print("[Exploration] Task ended unexpectedly during move to anchor position.")
                return
        except Exception as e:
            print("[Error] Exception during move to anchor position:", e)
            return

        # --- Step 4: Pick (grasp) the drawer handle ---
        print("[Exploration] Attempting to pick (grasp) the drawer handle.")
        try:
            obs, reward, done = pick(env, task, drawer_name, anchor_pos)
            if done:
                print("[Exploration] Task ended unexpectedly during pick.")
                return
        except Exception as e:
            print("[Error] Exception during pick:", e)
            return

        # --- Step 5: Pull the drawer to open it ---
        print("[Exploration] Attempting to pull the drawer open.")
        try:
            obs, reward, done = pull(env, task, drawer_name)
            if done:
                print("[Exploration] Task ended unexpectedly during pull.")
                return
        except Exception as e:
            print("[Error] Exception during pull:", e)
            return

        # --- Step 6: Check if the drawer is open (predicate is-open) ---
        # This is the exploration result: after pull, the predicate (is-open drawer0) should be true.
        print("[Exploration] Exploration complete. The predicate (is-open drawer0) should now be true if pull succeeded.")

        # --- (Optional) Additional task logic can be added here ---

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()