# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try to Discover Missing Predicate ===
        # The feedback suggests a missing predicate (e.g., lock-known, identified, temperature-known, etc.)
        # We'll attempt to use available skills to interact with objects and observe any unexpected behavior or errors.
        # This phase is to help identify which predicate is missing by performing basic actions and checking for issues.

        # List of available skills
        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']

        # Try to interact with all objects in the environment
        # For demonstration, we attempt to pick and place each object, and try pull/rotate if applicable
        for obj_name, obj_pos in positions.items():
            print(f"[Exploration] Attempting to interact with object: {obj_name} at {obj_pos}")
            try:
                # 1. Move to the object's position
                print(f"[Exploration] Moving to {obj_name} ...")
                obs, reward, done = move(env, task, target_pos=np.array(obj_pos))
                if done:
                    print(f"[Exploration] Task ended during move to {obj_name}!")
                    return

                # 2. Try to pick the object
                print(f"[Exploration] Trying to pick {obj_name} ...")
                try:
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after picking {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Exception during pick({obj_name}): {e}")

                # 3. Try to pull (if the object is a drawer or similar)
                if 'drawer' in obj_name.lower() or 'handle' in obj_name.lower():
                    print(f"[Exploration] Trying to pull {obj_name} ...")
                    try:
                        obs, reward, done = pull(
                            env,
                            task,
                            target_pos=np.array(obj_pos),
                            max_steps=100,
                            threshold=0.01,
                            timeout=10.0
                        )
                        if done:
                            print(f"[Exploration] Task ended after pulling {obj_name}!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Exception during pull({obj_name}): {e}")

                # 4. Try to rotate (if the object is a gripper or has an angle)
                if 'gripper' in obj_name.lower() or 'angle' in obj_name.lower():
                    print(f"[Exploration] Trying to rotate {obj_name} ...")
                    try:
                        # For demonstration, rotate from 0 to 90 degrees (if possible)
                        obs, reward, done = rotate(
                            env,
                            task,
                            gripper_name=obj_name,
                            from_angle='zero_deg',
                            to_angle='ninety_deg'
                        )
                        if done:
                            print(f"[Exploration] Task ended after rotating {obj_name}!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Exception during rotate({obj_name}): {e}")

                # 5. Try to place the object at its original position (or a default location)
                print(f"[Exploration] Trying to place {obj_name} ...")
                try:
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after placing {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Exception during place({obj_name}): {e}")

            except Exception as e:
                print(f"[Exploration] General exception for {obj_name}: {e}")

        print("[Exploration] Completed exploration phase. Check logs for any missing predicate clues.")

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this code, the focus is on exploration to identify missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
