# run_skeleton_task.py (Completed with Exploration Phase and Object List Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()  # Dictionary: object_name -> position

        # === Ensure All Required Objects Are Present ===
        # Feedback: Drawer objects may be missing from the object list.
        # We'll check for required objects and print warnings if missing.
        required_objects = ['drawer', 'gripper', 'bottom', 'handle']  # Example names; adjust as needed
        missing_objects = []
        for obj in required_objects:
            if obj not in positions:
                print(f"[Warning] Required object '{obj}' not found in object positions.")
                missing_objects.append(obj)
        if missing_objects:
            print("[Error] Missing required objects in the environment. Please check object list and initialization.")
            # Optionally, you could return here, but for exploration, continue.

        # === Exploration Phase: Identify Missing Predicate ===
        # The goal is to discover which predicate is missing (e.g., lock-known, durability-known, etc.)
        # We'll use available skills to interact with objects and observe state changes.

        # Example: Try to pick up all objects and see if any predicate is not updated as expected.
        # We'll log the predicates we can observe after each action.

        # For demonstration, let's iterate over all objects and try pick/place if possible.
        for obj_name, obj_pos in positions.items():
            print(f"[Exploration] Attempting to pick object '{obj_name}' at position {obj_pos}...")
            try:
                # Try to pick the object
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Picked '{obj_name}'. Observing predicates/state...")
                # After pick, check for known predicates (simulate: in real code, would check obs or task state)
                # For now, just print that we would check for 'weight-known', 'durability-known', etc.
                print(f"[Exploration] (Simulated) Checking if 'weight-known' and 'durability-known' for '{obj_name}'...")

                # Try to place the object back (if place skill is available)
                if 'place' in globals():
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Exploration] Placed '{obj_name}' back.")
            except Exception as e:
                print(f"[Exploration] Exception during pick/place of '{obj_name}': {e}")

        # Try to rotate and pull drawers if present
        for obj_name in positions:
            if 'drawer' in obj_name or 'handle' in obj_name:
                print(f"[Exploration] Attempting to rotate and pull '{obj_name}'...")
                try:
                    # Example: rotate the gripper to a target orientation (identity quaternion)
                    target_quat = np.array([0, 0, 0, 1])
                    obs, reward, done = rotate(
                        env,
                        task,
                        target_quat=target_quat,
                        max_steps=50,
                        threshold=0.05,
                        timeout=5.0
                    )
                    print(f"[Exploration] Rotated gripper for '{obj_name}'.")
                    # Try to pull (if pull skill is available)
                    if 'pull' in globals():
                        obs, reward, done = pull(
                            env,
                            task,
                            target_obj=obj_name,
                            max_steps=50,
                            threshold=0.01,
                            timeout=5.0
                        )
                        print(f"[Exploration] Pulled '{obj_name}'.")
                except Exception as e:
                    print(f"[Exploration] Exception during rotate/pull of '{obj_name}': {e}")

        # === Exploration Summary ===
        print("[Exploration] Exploration phase complete. Please review logs for missing predicates or object issues.")

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here using the available skills.
        # For this code, the focus is on exploration and predicate discovery as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
