# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try to discover missing predicates ===
        # The feedback and exploration knowledge suggest that some predicates (e.g., lock-known, identified, temperature-known, weight-known, durability-known) may be missing.
        # We will attempt to use available skills to interact with objects and observe any unexpected behavior or errors, which may indicate missing predicates.

        # For demonstration, we will attempt to:
        # 1. Move to each object position (if possible)
        # 2. Pick each object (if possible)
        # 3. Place each object (if possible)
        # 4. Try to rotate or pull if the object is a drawer

        # This exploration will help us identify which predicates or state transitions are not being handled.

        # List of available skills
        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']

        # For this exploration, we assume positions dict contains all relevant objects and their positions.
        # We'll iterate through them and attempt actions.

        for obj_name, obj_pos in positions.items():
            print(f"\n[Exploration] Attempting to interact with: {obj_name} at {obj_pos}")

            # 1. Move to the object's position (if 'move' is available)
            if 'move' in available_skills:
                try:
                    print(f"[Exploration] Moving to {obj_name} position...")
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during move to {obj_name}!")
                        break
                except Exception as e:
                    print(f"[Exploration] Exception during move to {obj_name}: {e}")

            # 2. Try to pick the object (if 'pick' is available)
            if 'pick' in available_skills:
                try:
                    print(f"[Exploration] Attempting to pick {obj_name}...")
                    # For pick, we may need to approach from above (z-axis), so offset z if needed
                    approach_pos = np.array(obj_pos) + np.array([0, 0, 0.15])
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=approach_pos,
                        max_steps=50,
                        threshold=0.01,
                        timeout=5.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during approach to pick {obj_name}!")
                        break
                    # Now move down to the object
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        max_steps=30,
                        threshold=0.005,
                        timeout=3.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during descend to pick {obj_name}!")
                        break
                    # Now pick
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        approach_distance=0.15,
                        max_steps=30,
                        threshold=0.005,
                        approach_axis='z',
                        timeout=5.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after picking {obj_name}!")
                        break
                except Exception as e:
                    print(f"[Exploration] Exception during pick of {obj_name}: {e}")

            # 3. Try to place the object at a default location (if 'place' is available)
            if 'place' in available_skills:
                try:
                    # Define a default place position (e.g., offset in x)
                    place_pos = np.array(obj_pos) + np.array([0.2, 0, 0])
                    print(f"[Exploration] Attempting to place {obj_name} at {place_pos}...")
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=place_pos + np.array([0, 0, 0.15]),
                        max_steps=50,
                        threshold=0.01,
                        timeout=5.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during approach to place {obj_name}!")
                        break
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=place_pos,
                        max_steps=30,
                        threshold=0.005,
                        timeout=3.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during descend to place {obj_name}!")
                        break
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=place_pos,
                        approach_distance=0.15,
                        max_steps=30,
                        threshold=0.005,
                        approach_axis='z',
                        timeout=5.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after placing {obj_name}!")
                        break
                except Exception as e:
                    print(f"[Exploration] Exception during place of {obj_name}: {e}")

            # 4. If the object is a drawer, try rotate and pull
            if ('rotate' in available_skills or 'pull' in available_skills) and 'drawer' in obj_name.lower():
                # For demonstration, try rotate and pull
                try:
                    if 'rotate' in available_skills:
                        print(f"[Exploration] Attempting to rotate {obj_name}...")
                        # For rotate, we need gripper, from_angle, to_angle
                        # These would be determined from the environment; here we use placeholders
                        gripper_name = 'gripper'
                        from_angle = 'zero_deg'
                        to_angle = 'ninety_deg'
                        obs, reward, done = rotate(
                            env,
                            task,
                            gripper_name,
                            from_angle,
                            to_angle
                        )
                        if done:
                            print(f"[Exploration] Task ended after rotating {obj_name}!")
                            break
                    if 'pull' in available_skills:
                        print(f"[Exploration] Attempting to pull {obj_name}...")
                        gripper_name = 'gripper'
                        obs, reward, done = pull(
                            env,
                            task,
                            gripper_name,
                            obj_name
                        )
                        if done:
                            print(f"[Exploration] Task ended after pulling {obj_name}!")
                            break
                except Exception as e:
                    print(f"[Exploration] Exception during rotate/pull of {obj_name}: {e}")

        print("\n[Exploration] Completed exploration phase. Check logs for any errors or unexpected terminations, which may indicate missing predicates or required state knowledge.")

        # === End of Exploration ===

        # After exploration, you would analyze the logs to determine which predicates are missing or which actions failed due to missing state knowledge.
        # This information can then be used to update the domain/problem or inform further debugging.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()