# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try to Discover Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will attempt to use available skills to interact with objects and observe the environment,
        # to help identify which predicate is missing (e.g., lock-known, identified, etc.)

        # List all objects and their positions
        print("[Exploration] Known objects and positions:")
        for obj_name, obj_pos in positions.items():
            print(f"    {obj_name}: {obj_pos}")

        # Try to move to each object and pick/place it, observing for errors or missing effects
        for obj_name, obj_pos in positions.items():
            print(f"\n[Exploration] Attempting to move to and pick/place: {obj_name}")

            try:
                # 1. Move to the object's position
                print(f"    [Action] move to {obj_name} at {obj_pos}")
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"    [Result] Task ended during move to {obj_name}.")
                    break

                # 2. Try to pick the object
                print(f"    [Action] pick {obj_name}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"    [Result] Task ended during pick of {obj_name}.")
                    break

                # 3. Try to place the object back at its original position (or a default location)
                print(f"    [Action] place {obj_name} at {obj_pos}")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"    [Result] Task ended during place of {obj_name}.")
                    break

            except Exception as e:
                print(f"    [Error] Exception during exploration of {obj_name}: {e}")
                continue

        # Try to rotate and pull any drawers if present
        for obj_name, obj_pos in positions.items():
            if "drawer" in obj_name.lower():
                print(f"\n[Exploration] Attempting to rotate and pull: {obj_name}")
                try:
                    # For demonstration, try rotating the gripper to 90 degrees (if required)
                    print(f"    [Action] rotate gripper for {obj_name}")
                    obs, reward, done = rotate(
                        env,
                        task,
                        gripper_name="gripper",  # Replace with actual gripper name if available
                        from_angle="zero_deg",
                        to_angle="ninety_deg"
                    )
                    if done:
                        print(f"    [Result] Task ended during rotate for {obj_name}.")
                        break

                    # Try to pull the drawer
                    print(f"    [Action] pull {obj_name}")
                    obs, reward, done = pull(
                        env,
                        task,
                        drawer_name=obj_name
                    )
                    if done:
                        print(f"    [Result] Task ended during pull of {obj_name}.")
                        break

                except Exception as e:
                    print(f"    [Error] Exception during rotate/pull of {obj_name}: {e}")
                    continue

        print("\n[Exploration] Completed. If errors or unexpected terminations occurred, check logs for clues about missing predicates or required knowledge (e.g., lock-known, identified, etc.).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
