# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'item1': (x, y, z), 'table': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (at item1 table) is missing, so we need to check if the robot can perceive/identify item1 at table.
        # According to exploration knowledge, moving to a location allows the robot to "identify" objects at that location.
        # We'll simulate this by moving to the table and checking for item1.

        # Assume 'item1' and 'table' are keys in positions
        try:
            item1_pos = positions['item1']
            table_pos = positions['table']
        except KeyError as e:
            print(f"[Error] Missing object in positions: {e}")
            shutdown_environment(env)
            return

        # 1. Move to the table to "identify" item1 (exploration)
        print("[Exploration] Moving to table to identify objects present (expecting item1).")
        try:
            # The 'move' skill is assumed to move the robot to a target position.
            # Since the skill signature is not provided, we assume: move(env, task, target_pos, ...)
            obs, reward, done = move(
                env,
                task,
                target_pos=table_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during move to table!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to table: {e}")
            shutdown_environment(env)
            return

        # 2. Check if item1 is at the table (simulate perception)
        # In a real system, this would involve perception; here, we use the feedback and positions.
        print("[Exploration] Checking for missing predicate: (at item1 table)")
        if 'item1' in positions and 'table' in positions:
            # If item1 is close to table, we consider (at item1 table) to be true
            dist = np.linalg.norm(np.array(item1_pos) - np.array(table_pos))
            if dist < 0.2:  # Threshold for "at"
                print("[Exploration] Predicate found: (at item1 table) is TRUE.")
            else:
                print("[Exploration] Predicate (at item1 table) is FALSE or missing.")
        else:
            print("[Exploration] item1 or table not found in positions.")

        # 3. Now, execute the plan: pick item1 from the table
        print("[Task] Picking item1 from table.")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=item1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick: {e}")
            shutdown_environment(env)
            return

        # 4. (Optional) Place item1 somewhere else if required by the plan
        # For demonstration, let's place item1 back on the table (or another location if specified)
        print("[Task] Placing item1 back on table.")
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=table_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after place!")
                return
        except Exception as e:
            print(f"[Error] Exception during place: {e}")
            shutdown_environment(env)
            return

        print("[Task] Exploration and manipulation complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
