# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will perform an exploration phase to try to discover which predicate is missing.
        # This is done by attempting to use the available skills on all objects/positions and
        # observing if any action fails due to a missing precondition or effect.

        # List of available skills
        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']

        # For exploration, we will attempt to use each skill on each object/position
        # and catch any exceptions or failures that may indicate a missing predicate.

        # Keep track of which predicates might be missing
        missing_predicates = set()

        # Get all object names and their positions
        object_names = list(positions.keys())

        # Try each skill in a safe way
        for skill in available_skills:
            for obj_name in object_names:
                obj_pos = positions[obj_name]
                try:
                    print(f"[Exploration] Trying skill '{skill}' on object '{obj_name}' at position {obj_pos}")
                    if skill == 'move':
                        # Try moving to the object's position
                        obs, reward, done = move(env, task, target_pos=np.array(obj_pos))
                        if done:
                            print("[Exploration] Task ended during move.")
                            break
                    elif skill == 'pick':
                        # Try picking the object
                        # Use a default approach distance and axis
                        obs, reward, done = pick(
                            env,
                            task,
                            target_pos=np.array(obj_pos),
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        if done:
                            print("[Exploration] Task ended during pick.")
                            break
                    elif skill == 'place':
                        # Try placing the object at its current position (as a test)
                        obs, reward, done = place(
                            env,
                            task,
                            target_pos=np.array(obj_pos),
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        if done:
                            print("[Exploration] Task ended during place.")
                            break
                    elif skill == 'rotate':
                        # Try rotating the gripper (if possible)
                        # For demonstration, try rotating from 0 to 90 degrees (in radians)
                        # This assumes the rotate skill takes gripper, from_angle, to_angle
                        # We'll use dummy values as placeholders
                        gripper_name = 'gripper'
                        from_angle = 0.0
                        to_angle = np.pi / 2
                        try:
                            obs, reward, done = rotate(
                                env,
                                task,
                                gripper=gripper_name,
                                from_angle=from_angle,
                                to_angle=to_angle
                            )
                        except Exception as e:
                            print(f"[Exploration] Rotate failed: {e}")
                            missing_predicates.add('rotated')
                        if done:
                            print("[Exploration] Task ended during rotate.")
                            break
                    elif skill == 'pull':
                        # Try pulling the object (if possible)
                        # This assumes the pull skill takes the object name
                        try:
                            obs, reward, done = pull(
                                env,
                                task,
                                obj_name
                            )
                        except Exception as e:
                            print(f"[Exploration] Pull failed: {e}")
                            missing_predicates.add('is-locked')
                        if done:
                            print("[Exploration] Task ended during pull.")
                            break
                except Exception as e:
                    print(f"[Exploration] Skill '{skill}' failed on '{obj_name}': {e}")
                    # Based on the error message, try to infer the missing predicate
                    if 'handempty' in str(e):
                        missing_predicates.add('handempty')
                    if 'at' in str(e):
                        missing_predicates.add('at')
                    if 'holding' in str(e):
                        missing_predicates.add('holding')
                    if 'rotated' in str(e):
                        missing_predicates.add('rotated')
                    if 'is-locked' in str(e):
                        missing_predicates.add('is-locked')
                    if 'is-open' in str(e):
                        missing_predicates.add('is-open')
                    if 'gripper-at' in str(e):
                        missing_predicates.add('gripper-at')
                    if 'holding-drawer' in str(e):
                        missing_predicates.add('holding-drawer')
                    if 'is-side-pos' in str(e):
                        missing_predicates.add('is-side-pos')
                    if 'is-anchor-pos' in str(e):
                        missing_predicates.add('is-anchor-pos')
                    # Add more as needed based on domain predicates

        if missing_predicates:
            print("[Exploration] Possible missing predicates detected:")
            for pred in missing_predicates:
                print(f"    - {pred}")
        else:
            print("[Exploration] No missing predicates detected during skill execution.")

        # === End of Exploration Phase ===

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this code, we focus on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()