# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Try to discover missing predicates ===
        # The feedback suggests that a predicate required for planning is missing.
        # We will attempt to use the available skills to interact with objects and locations,
        # and observe if any information (e.g., lock status, temperature, durability, etc.) is revealed.

        # For demonstration, we will:
        # 1. Move to each location (if possible)
        # 2. Attempt to pick each object (if possible)
        # 3. Attempt to pull (if possible)
        # 4. Rotate (if possible)
        # 5. Place (if possible)
        # This will help us discover which predicates are missing by observing the environment's response.

        # Get all objects and locations from positions
        object_names = []
        location_names = []
        for name, info in positions.items():
            if info.get('type', '') == 'object':
                object_names.append(name)
            elif info.get('type', '') == 'location':
                location_names.append(name)
        # Fallback: if no type info, try to infer from name
        if not object_names or not location_names:
            for name in positions.keys():
                if 'drawer' in name or 'obj' in name or 'ball' in name or 'item' in name:
                    if name not in object_names:
                        object_names.append(name)
                if 'room' in name or 'loc' in name or 'bin' in name or 'table' in name:
                    if name not in location_names:
                        location_names.append(name)

        # Try to move to each location (if move skill is available)
        if 'move' in ['pick', 'place', 'move', 'rotate', 'pull']:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to move to location: {loc}")
                    obs, reward, done = move(env, task, target_location=loc)
                    if done:
                        print(f"[Exploration] Task ended after move to {loc}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Move to {loc} failed: {e}")

        # Try to pick each object at each location (if pick skill is available)
        if 'pick' in ['pick', 'place', 'move', 'rotate', 'pull']:
            for obj in object_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting to pick object: {obj} at location: {loc}")
                        obs, reward, done = pick(env, task, obj, loc)
                        if done:
                            print(f"[Exploration] Task ended after pick {obj} at {loc}!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Pick {obj} at {loc} failed: {e}")

        # Try to pull each object at each location (if pull skill is available)
        if 'pull' in ['pick', 'place', 'move', 'rotate', 'pull']:
            for obj in object_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting to pull object: {obj} at location: {loc}")
                        obs, reward, done = pull(env, task, obj, loc)
                        if done:
                            print(f"[Exploration] Task ended after pull {obj} at {loc}!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Pull {obj} at {loc} failed: {e}")

        # Try to rotate (if rotate skill is available)
        if 'rotate' in ['pick', 'place', 'move', 'rotate', 'pull']:
            # For demonstration, try rotating the gripper to a few standard quaternions
            from skill_code import normalize_quaternion, euler_from_quat
            # Try 0 deg and 90 deg (in radians)
            import math
            for angle_deg in [0, 90]:
                angle_rad = math.radians(angle_deg)
                # Convert to quaternion (z axis rotation)
                q = [0, 0, np.sin(angle_rad/2), np.cos(angle_rad/2)]
                try:
                    print(f"[Exploration] Attempting to rotate gripper to {angle_deg} degrees")
                    obs, reward, done = rotate(env, task, q)
                    if done:
                        print(f"[Exploration] Task ended after rotate to {angle_deg} deg!")
                        return
                except Exception as e:
                    print(f"[Exploration] Rotate to {angle_deg} deg failed: {e}")

        # Try to place each object at each location (if place skill is available)
        if 'place' in ['pick', 'place', 'move', 'rotate', 'pull']:
            for obj in object_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting to place object: {obj} at location: {loc}")
                        obs, reward, done = place(env, task, obj, loc)
                        if done:
                            print(f"[Exploration] Task ended after place {obj} at {loc}!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Place {obj} at {loc} failed: {e}")

        print("[Exploration] Completed exploration phase. Check logs for missing predicate clues.")

        # === (Optional) Main Task Plan ===
        # After exploration, you would normally execute the oracle plan here.
        # For this code, the focus is on exploration to discover missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()