# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try Each Skill to Find Missing Predicate ===
        # The goal is to explore the environment using available skills to discover which predicate is missing.
        # We will attempt to use each skill in a safe, exploratory way and observe the results.

        # List of available skills
        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']

        # For demonstration, we will attempt to use each skill on available objects/positions
        # and print the outcome. This helps identify which predicates are missing or not handled.

        # Get all object names and positions
        object_names = list(positions.keys())
        if not object_names:
            print("[Exploration] No objects found in the environment.")
        else:
            print(f"[Exploration] Found objects: {object_names}")

        # Try to move the gripper to each object's position
        for obj_name in object_names:
            target_pos = positions[obj_name]
            print(f"[Exploration] Attempting to move to {obj_name} at {target_pos}")
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(target_pos),
                    max_steps=50,
                    threshold=0.01,
                    timeout=5.0
                )
                if done:
                    print(f"[Exploration] Task ended after moving to {obj_name}.")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during move to {obj_name}: {e}")

        # Try to pick each object (if possible)
        for obj_name in object_names:
            target_pos = positions[obj_name]
            print(f"[Exploration] Attempting to pick {obj_name} at {target_pos}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(target_pos),
                    approach_distance=0.15,
                    max_steps=50,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}.")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pick of {obj_name}: {e}")

        # Try to place each object at its original position (if possible)
        for obj_name in object_names:
            target_pos = positions[obj_name]
            print(f"[Exploration] Attempting to place {obj_name} at {target_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(target_pos),
                    approach_distance=0.15,
                    max_steps=50,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=5.0
                )
                if done:
                    print(f"[Exploration] Task ended after placing {obj_name}.")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during place of {obj_name}: {e}")

        # Try to rotate the gripper (if possible)
        # For demonstration, try rotating between two angles if available
        angles = ['zero_deg', 'ninety_deg']
        gripper_name = None
        for name in object_names:
            if 'gripper' in name:
                gripper_name = name
                break
        if gripper_name:
            print(f"[Exploration] Attempting to rotate {gripper_name}")
            try:
                obs, reward, done = rotate(
                    env,
                    task,
                    gripper_name,
                    from_angle='zero_deg',
                    to_angle='ninety_deg'
                )
                if done:
                    print(f"[Exploration] Task ended after rotating {gripper_name}.")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during rotate of {gripper_name}: {e}")
        else:
            print("[Exploration] No gripper found for rotate action.")

        # Try to pull each object (if possible)
        for obj_name in object_names:
            print(f"[Exploration] Attempting to pull {obj_name}")
            try:
                obs, reward, done = pull(
                    env,
                    task,
                    obj_name
                )
                if done:
                    print(f"[Exploration] Task ended after pulling {obj_name}.")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pull of {obj_name}: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
