# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example: positions = {'drawer1': np.array([...]), 'gripper': np.array([...]), ...}

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (is-open drawer1) is missing or needs to be checked.
        # We will attempt to open the drawer and check if the predicate is achieved.

        # Assumptions for object names (adjust as needed based on your environment):
        drawer_name = 'drawer1'
        gripper_name = 'gripper'
        # Positions for the gripper to interact with the drawer
        # These should be defined in object_positions or inferred from the environment
        try:
            drawer_pos = positions[drawer_name]
            gripper_pos = positions[gripper_name]
        except KeyError as e:
            print(f"[Error] Missing object position for: {e}")
            shutdown_environment(env)
            return

        # 1. Move gripper to the side position of the drawer (preparation for rotation)
        print(f"[Exploration] Moving gripper to side position of {drawer_name}.")
        try:
            obs, reward, done = move(env, task, target_pos=gripper_pos)
            if done:
                print("[Exploration] Task ended unexpectedly during move to side position.")
                return
        except Exception as e:
            print(f"[Error] Exception during move: {e}")
            shutdown_environment(env)
            return

        # 2. Rotate gripper to required angle (e.g., 90 degrees) for drawer manipulation
        # The rotate skill may require angle names or values; adjust as needed.
        print(f"[Exploration] Rotating gripper for drawer manipulation.")
        try:
            # Example: rotate(env, task, gripper_name, from_angle, to_angle)
            # We'll use 'zero_deg' to 'ninety_deg' as per PDDL domain
            obs, reward, done = rotate(env, task, gripper_name, 'zero_deg', 'ninety_deg')
            if done:
                print("[Exploration] Task ended unexpectedly during rotation.")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # 3. Move gripper to anchor position (for picking the drawer handle)
        # Assume anchor position is provided or can be inferred
        anchor_pos_key = f"{drawer_name}_anchor"
        if anchor_pos_key in positions:
            anchor_pos = positions[anchor_pos_key]
        else:
            anchor_pos = drawer_pos  # Fallback if anchor not specified

        print(f"[Exploration] Moving gripper to anchor position of {drawer_name}.")
        try:
            obs, reward, done = move(env, task, target_pos=anchor_pos)
            if done:
                print("[Exploration] Task ended unexpectedly during move to anchor position.")
                return
        except Exception as e:
            print(f"[Error] Exception during move to anchor: {e}")
            shutdown_environment(env)
            return

        # 4. Pick the drawer handle (if required by the domain)
        # The pick skill may require object and location; adjust as needed.
        print(f"[Exploration] Picking the drawer handle.")
        try:
            obs, reward, done = pick(env, task, drawer_name, anchor_pos)
            if done:
                print("[Exploration] Task ended unexpectedly during pick.")
                return
        except Exception as e:
            print(f"[Error] Exception during pick: {e}")
            shutdown_environment(env)
            return

        # 5. Pull the drawer to open it
        print(f"[Exploration] Pulling the drawer to open it.")
        try:
            obs, reward, done = pull(env, task, drawer_name)
            if done:
                print("[Exploration] Task ended unexpectedly during pull.")
                return
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # 6. Check if the drawer is open (predicate: is-open drawer1)
        # This is the missing predicate from feedback; we can check the state or print a message.
        print(f"[Exploration] Checking if the drawer is open (is-open {drawer_name}).")
        # If the environment provides a way to check predicates, do so here.
        # Otherwise, rely on successful completion of the pull action.

        # === End of Exploration Phase ===

        print("[Exploration] Exploration phase complete. If the drawer is open, the missing predicate is (is-open drawer1).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()