# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to check for missing predicates such as force calibration, safety, and object presence.
        # We will attempt to pick an object and check for errors or unexpected behavior.

        # For demonstration, let's try to pick up the first available object in the environment.
        # This will help us identify if any predicate (e.g., weight-known, durability-known, or lock-known) is missing.

        # Find a pickable object and its position
        pickable_object = None
        pickable_object_pos = None
        for obj_name, obj_pos in positions.items():
            if "drawer" not in obj_name and "gripper" not in obj_name and "angle" not in obj_name and "position" not in obj_name:
                pickable_object = obj_name
                pickable_object_pos = obj_pos
                break

        if pickable_object is None or pickable_object_pos is None:
            print("[Exploration] No pickable object found in the environment.")
        else:
            print(f"[Exploration] Attempting to pick object: {pickable_object} at position {pickable_object_pos}")

            # 1. Move to above the object (safe approach)
            approach_offset = np.array([0, 0, 0.15])  # Approach from above
            approach_pos = np.array(pickable_object_pos) + approach_offset

            try:
                # Safety check: is_safe_position (if available)
                if 'is_safe_position' in globals():
                    if not is_safe_position(approach_pos):
                        print("[Exploration] Approach position is not safe. Aborting pick attempt.")
                        return

                obs, reward, done = move(
                    env,
                    task,
                    target_pos=approach_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended during approach move!")
                    return

                # 2. Move down to the object's position
                pick_pos = np.array(pickable_object_pos)
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=pick_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended during pick move!")
                    return

                # 3. Attempt to pick the object
                # Check for force calibration parameter in the skill (if available)
                pick_kwargs = dict(
                    env=env,
                    task=task,
                    target_pos=pick_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if 'force_calibration' in pick.__code__.co_varnames:
                    pick_kwargs['force_calibration'] = True

                obs, reward, done = pick(**pick_kwargs)
                if done:
                    print("[Exploration] Task ended after pick attempt!")
                    return

                # 4. Move up after picking
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=approach_pos,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after moving up post-pick!")
                    return

                print("[Exploration] Pick attempt completed. If errors occurred, check logs for missing predicates.")

            except Exception as e:
                print(f"[Exploration] Exception during pick attempt: {e}")
                print("[Exploration] This may indicate a missing predicate or skill precondition (e.g., force calibration, safety, object presence, collision, etc.).")
                return

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate, update the plan or inform the planner as needed.
        # For now, the code demonstrates how to explore and check for missing predicates using only available skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()