# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration PDDL suggest that the robot may need to discover a missing predicate,
        # such as 'lock-known' for a drawer or object. We will perform an exploration step using available skills.

        # For demonstration, let's assume we have a drawer object and a gripper, and we want to check if the drawer is locked.
        # We'll use the 'pull' skill to try to pull the drawer and observe the result, which may reveal the missing predicate.

        # Example object keys (replace with actual keys from your environment if different)
        drawer_name = None
        gripper_name = None
        drawer_pos = None

        # Try to find a drawer and gripper in the positions dictionary
        for obj_name in positions:
            if 'drawer' in obj_name.lower():
                drawer_name = obj_name
                drawer_pos = positions[obj_name]
            if 'gripper' in obj_name.lower():
                gripper_name = obj_name

        if drawer_name is None or gripper_name is None:
            print("[Exploration] Could not find drawer or gripper in object positions. Exploration skipped.")
        else:
            print(f"[Exploration] Attempting to pull drawer '{drawer_name}' with gripper '{gripper_name}'.")

            # 1. Rotate gripper to appropriate angle if needed (e.g., 'ninety_deg')
            #    For demonstration, we use a placeholder quaternion for 90 degrees rotation.
            #    In practice, retrieve the correct quaternion for 'ninety_deg' from your environment.
            try:
                target_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])  # 90 deg about z
                obs, reward, done = rotate(env, task, target_quat)
                if done:
                    print("[Exploration] Task ended during rotation!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during rotate: {e}")

            # 2. Attempt to pull the drawer
            try:
                obs, reward, done = pull(env, task, drawer_name)
                if done:
                    print("[Exploration] Task ended during pull!")
                    return
                print("[Exploration] Pull action executed. Check if lock-known predicate is now revealed in the state.")
            except Exception as e:
                print(f"[Exploration] Exception during pull: {e}")

        # === Main Task Plan (Example) ===
        # After exploration, proceed with the main plan using available skills.
        # For demonstration, let's try to pick and place an object if available.

        # Find an object to pick (not a drawer or gripper)
        pick_obj_name = None
        pick_obj_pos = None
        for obj_name in positions:
            if 'drawer' not in obj_name.lower() and 'gripper' not in obj_name.lower():
                pick_obj_name = obj_name
                pick_obj_pos = positions[obj_name]
                break

        if pick_obj_name is not None and pick_obj_pos is not None:
            print(f"[Task] Attempting to pick object '{pick_obj_name}' at position {pick_obj_pos}.")
            try:
                obs, reward, done = pick(env, task, pick_obj_name, pick_obj_pos)
                if done:
                    print("[Task] Task ended after pick!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pick: {e}")

            # Place the object at a target location (for demonstration, offset the position)
            place_pos = np.array(pick_obj_pos) + np.array([0.1, 0.0, 0.0])
            print(f"[Task] Attempting to place object '{pick_obj_name}' at position {place_pos}.")
            try:
                obs, reward, done = place(env, task, pick_obj_name, place_pos)
                if done:
                    print("[Task] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Task] Exception during place: {e}")
        else:
            print("[Task] No suitable object found for pick and place.")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()