# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x,y,z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (at gripper nowhere-pos)
        # The feedback suggests that the predicate (at gripper nowhere-pos) is missing or not handled.
        # We will perform an exploration step to check the gripper's position and log the relevant predicate.

        print("=== [Exploration] Checking for missing predicate: (at gripper nowhere-pos) ===")
        try:
            obs = task.get_observation()
            gripper_pos = obs.gripper_pose[:3] if hasattr(obs, 'gripper_pose') else None
            if gripper_pos is not None:
                print(f"[Exploration] Gripper position: {gripper_pos}")
                # Check if gripper is at a known position or 'nowhere'
                found = False
                for pos_name, pos_val in positions.items():
                    if np.allclose(gripper_pos, pos_val, atol=0.05):
                        print(f"[Exploration] Gripper is at: {pos_name}")
                        found = True
                        break
                if not found:
                    print("[Exploration] Gripper is at an unknown position (nowhere-pos).")
                    print("[Exploration] Missing predicate detected: (at gripper nowhere-pos)")
            else:
                print("[Exploration] Could not retrieve gripper position from observation.")
        except Exception as e:
            print(f"[Exploration] Exception during exploration: {e}")

        # === Example Task Plan ===
        # For demonstration, we will attempt to pick and place an object if available.
        # Replace 'object_1' and 'target_location' with actual keys from positions as needed.

        # Find an object to pick and a target location to place
        object_to_pick = None
        target_location = None
        for name in positions:
            if 'object' in name or 'ball' in name or 'cube' in name:
                object_to_pick = name
                break
        for name in positions:
            if 'target' in name or 'goal' in name or 'bin' in name or 'place' in name:
                target_location = name
                break

        if object_to_pick is not None:
            print(f"[Task] Attempting to pick: {object_to_pick} at {positions[object_to_pick]}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(positions[object_to_pick]),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after pick!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pick: {e}")
        else:
            print("[Task] No pickable object found in positions.")

        if object_to_pick is not None and target_location is not None:
            print(f"[Task] Attempting to place: {object_to_pick} at {positions[target_location]}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(positions[target_location]),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Task] Exception during place: {e}")
        else:
            print("[Task] No valid target location found for place.")

        # === End of Task Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
