# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'bottom_drawer': ..., 'gripper': ..., ...}
        positions = get_object_positions()

        # --- Exploration Phase: Check for missing predicate (is-open bottom) ---
        # The feedback indicates that the predicate (is-open bottom) is missing or not achieved.
        # We need to explore the environment to determine if the bottom drawer is open.
        # We'll use the available skills to try to open the bottom drawer.

        # Assumptions for object names (update as needed based on your environment):
        # - The bottom drawer is named 'bottom' or 'bottom_drawer'
        # - The gripper is named 'gripper' or similar
        # - Positions dict contains keys for relevant positions

        # Try to robustly get the names/positions
        drawer_name = None
        for key in positions.keys():
            if 'bottom' in key and 'drawer' in key:
                drawer_name = key
                break
            elif key == 'bottom':
                drawer_name = key
        if drawer_name is None:
            print("[Error] Could not find bottom drawer in object positions.")
            return

        gripper_name = None
        for key in positions.keys():
            if 'gripper' in key:
                gripper_name = key
                break
        if gripper_name is None:
            print("[Error] Could not find gripper in object positions.")
            return

        # Get positions for gripper and drawer
        gripper_pos = positions[gripper_name]
        drawer_pos = positions[drawer_name]

        # For the drawer, we may need to know anchor and side positions, and angles
        # We'll try to infer these from the positions dict
        anchor_pos = None
        side_pos = None
        for key in positions.keys():
            if 'anchor' in key and drawer_name in key:
                anchor_pos = positions[key]
            if 'side' in key and drawer_name in key:
                side_pos = positions[key]
        # Fallback: use drawer_pos as anchor if not found
        if anchor_pos is None:
            anchor_pos = drawer_pos
        if side_pos is None:
            side_pos = drawer_pos

        # Angles: assume 'zero_deg' and 'ninety_deg' are available as constants or in positions
        zero_deg = None
        ninety_deg = None
        for key in positions.keys():
            if 'zero_deg' in key:
                zero_deg = positions[key]
            if 'ninety_deg' in key:
                ninety_deg = positions[key]
        # If not found, set to 0 and 90 (radians)
        if zero_deg is None:
            zero_deg = 0.0
        if ninety_deg is None:
            ninety_deg = np.pi / 2

        # --- Plan to open the bottom drawer (exploration for is-open bottom) ---
        # 1. Move gripper to side position (if needed)
        # 2. Rotate gripper to 90 degrees (if needed)
        # 3. Move gripper to anchor position
        # 4. Pick the drawer handle (pick-drawer)
        # 5. Pull the drawer open (pull)
        # Only use available skills: move, pick, place, rotate, pull

        # Step 1: Move gripper to side position (if not already there)
        print("[Exploration] Moving gripper to side position of bottom drawer.")
        try:
            obs, reward, done = move(env, task, target_pos=side_pos)
            if done:
                print("[Exploration] Task ended unexpectedly during move to side position.")
                return
        except Exception as e:
            print(f"[Error] Exception during move to side position: {e}")
            return

        # Step 2: Rotate gripper to 90 degrees
        print("[Exploration] Rotating gripper to 90 degrees.")
        try:
            obs = task.get_observation()
            # Assume we can get current angle from obs, else skip
            current_angle = getattr(obs, 'gripper_angle', zero_deg)
            if current_angle != ninety_deg:
                obs, reward, done = rotate(env, task, gripper_name, current_angle, ninety_deg)
                if done:
                    print("[Exploration] Task ended unexpectedly during rotate.")
                    return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            return

        # Step 3: Move gripper to anchor position
        print("[Exploration] Moving gripper to anchor position of bottom drawer.")
        try:
            obs, reward, done = move(env, task, target_pos=anchor_pos)
            if done:
                print("[Exploration] Task ended unexpectedly during move to anchor position.")
                return
        except Exception as e:
            print(f"[Error] Exception during move to anchor position: {e}")
            return

        # Step 4: Pull the drawer open
        print("[Exploration] Pulling the bottom drawer open.")
        try:
            obs, reward, done = pull(env, task, gripper_name, drawer_name)
            if done:
                print("[Exploration] Task ended unexpectedly during pull.")
                return
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            return

        print("[Exploration] Finished exploration for is-open bottom predicate.")

        # --- End of Exploration Phase ---

        # At this point, the predicate (is-open bottom) should be true in the environment.
        # You can now proceed with further task logic if needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()