# run_skeleton_task.py (Completed with Exploration Phase and Robustness)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

import traceback

def log_error(msg):
    print(f"[ERROR] {msg}")

def is_object_reachable(gripper_pos, obj_pos, max_reach=0.5):
    """Check if the object is within the gripper's reach."""
    dist = np.linalg.norm(np.array(gripper_pos) - np.array(obj_pos))
    return dist <= max_reach

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Object List Validation ===
        # Only use objects that are actually present in the environment
        available_objects = list(positions.keys())
        print("[Info] Available objects in environment:", available_objects)

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that some predicates (e.g., lock-known, identified, temperature-known, etc.)
        # may be missing or not properly handled. We will perform an exploration step to check for missing predicates
        # by attempting to interact with all objects and locations using only available skills.

        # For demonstration, we will try to move to each object's location (if move is available), pick it (if possible), and place it back.
        # This will help us discover if any predicate (e.g., identified, weight-known, durability-known, lock-known) is missing in the environment.

        # Safety: Check if the gripper can reach the object before attempting pick/place.
        # Error handling: Use try-except to catch and log any issues.

        # Get initial gripper position (if available)
        try:
            obs = task.get_observation()
            gripper_pose = getattr(obs, 'gripper_pose', None)
            if gripper_pose is not None:
                gripper_pos = gripper_pose[:3]
            else:
                gripper_pos = np.zeros(3)
        except Exception as e:
            log_error(f"Failed to get initial gripper position: {e}")
            gripper_pos = np.zeros(3)

        # Main exploration loop
        for obj_name in available_objects:
            obj_pos = positions[obj_name]
            print(f"\n[Exploration] Attempting to interact with object: {obj_name} at {obj_pos}")

            # 1. Move to object (if move skill is available)
            if 'move' in ['pick', 'place', 'move', 'rotate', 'pull']:
                try:
                    # For demonstration, we assume move(env, task, target_pos, ...) signature
                    print(f"[Exploration] Moving to {obj_name} at {obj_pos}")
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during move to {obj_name}!")
                        break
                except Exception as e:
                    log_error(f"Move to {obj_name} failed: {e}")
                    traceback.print_exc()
                    continue

            # 2. Safety check: Is object reachable?
            try:
                obs = task.get_observation()
                gripper_pose = getattr(obs, 'gripper_pose', None)
                if gripper_pose is not None:
                    gripper_pos = gripper_pose[:3]
                else:
                    gripper_pos = np.zeros(3)
                if not is_object_reachable(gripper_pos, obj_pos):
                    log_error(f"Object {obj_name} is not reachable by the gripper. Skipping pick.")
                    continue
            except Exception as e:
                log_error(f"Failed to check reachability for {obj_name}: {e}")
                continue

            # 3. Pick object (if pick skill is available)
            if 'pick' in ['pick', 'place', 'move', 'rotate', 'pull']:
                try:
                    print(f"[Exploration] Picking {obj_name}")
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during pick of {obj_name}!")
                        break
                except Exception as e:
                    log_error(f"Pick {obj_name} failed: {e}")
                    traceback.print_exc()
                    continue

            # 4. Place object back (if place skill is available)
            if 'place' in ['pick', 'place', 'move', 'rotate', 'pull']:
                try:
                    print(f"[Exploration] Placing {obj_name} back at {obj_pos}")
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during place of {obj_name}!")
                        break
                except Exception as e:
                    log_error(f"Place {obj_name} failed: {e}")
                    traceback.print_exc()
                    continue

            # 5. Try to pull (if pull skill is available and object is a drawer/handle)
            if 'pull' in ['pick', 'place', 'move', 'rotate', 'pull']:
                # Heuristic: Try pull only for objects with 'drawer' or 'handle' in their name
                if 'drawer' in obj_name or 'handle' in obj_name:
                    try:
                        print(f"[Exploration] Attempting to pull {obj_name}")
                        obs, reward, done = pull(
                            env,
                            task,
                            target_pos=obj_pos,
                            approach_distance=0.10,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        if done:
                            print(f"[Exploration] Task ended during pull of {obj_name}!")
                            break
                    except Exception as e:
                        log_error(f"Pull {obj_name} failed: {e}")
                        traceback.print_exc()
                        continue

            # 6. Try to rotate (if rotate skill is available and object is a handle/rotatable)
            if 'rotate' in ['pick', 'place', 'move', 'rotate', 'pull']:
                if 'handle' in obj_name or 'knob' in obj_name:
                    try:
                        print(f"[Exploration] Attempting to rotate {obj_name}")
                        # For demonstration, use a fixed target quaternion (identity)
                        target_quat = np.array([0, 0, 0, 1])
                        obs, reward, done = rotate(
                            env,
                            task,
                            target_quat=target_quat,
                            max_steps=100,
                            threshold=0.05,
                            timeout=10.0
                        )
                        if done:
                            print(f"[Exploration] Task ended during rotate of {obj_name}!")
                            break
                    except Exception as e:
                        log_error(f"Rotate {obj_name} failed: {e}")
                        traceback.print_exc()
                        continue

        print("\n[Exploration] Completed interaction with all available objects.")

        # === End of Exploration Phase ===
        # At this point, if any predicate is missing (e.g., lock-known, identified, etc.), the above steps and their logs
        # should help identify which skill or predicate is not being triggered or is missing in the environment.

        # === (Optional) Main Task Plan Execution ===
        # If you have a specific oracle plan, you can implement it here using the same skill calls as above,
        # ensuring to use only the correct object names and positions from the environment.

    except Exception as main_e:
        log_error(f"Exception in main task: {main_e}")
        traceback.print_exc()
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
