# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (is-side-pos plate_above ?d) is missing.
        # We need to explore the environment to check if 'plate_above' is a valid side position for any drawer (?d).
        # Since we cannot directly check predicates, we attempt to move the gripper to the 'plate_above' position
        # and observe if the environment allows us to perform actions that require (is-side-pos plate_above ?d).
        # This exploration is necessary to determine if the missing predicate is required for the plan.

        # Attempt to find 'plate_above' and a drawer in the environment
        plate_above_pos = None
        drawer_name = None
        for key in positions:
            if 'plate_above' in key:
                plate_above_pos = positions[key]
            if 'drawer' in key and drawer_name is None:
                drawer_name = key  # Take the first drawer found

        if plate_above_pos is None or drawer_name is None:
            print("[Exploration] Could not find 'plate_above' or a drawer in object positions.")
        else:
            print(f"[Exploration] Found plate_above at {plate_above_pos}, drawer: {drawer_name}")

            # Try to move the gripper to the plate_above position
            try:
                print("[Exploration] Moving gripper to plate_above position to test (is-side-pos plate_above ?d)...")
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(plate_above_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print("[Exploration] Successfully moved to plate_above position.")
                # If the move is successful, we can infer that (is-side-pos plate_above ?d) is likely valid.
                # Otherwise, the environment may not allow this, indicating the predicate is missing.
            except Exception as e:
                print(f"[Exploration] Failed to move to plate_above: {e}")
                print("[Exploration] This suggests (is-side-pos plate_above ?d) may be missing or invalid.")

        # === Main Task Plan Execution ===
        # The following is a generic plan using only the available skills: ['pick', 'place', 'move', 'rotate', 'pull']
        # You should adapt this section to your specific oracle plan and environment.

        # Example: Pick up an object and place it somewhere else
        # For demonstration, we try to pick the first object (not a drawer) and place it at a target location

        # Find a pickable object and a target location
        pick_obj = None
        pick_obj_pos = None
        place_loc = None
        for key in positions:
            if 'drawer' not in key and 'plate_above' not in key:
                pick_obj = key
                pick_obj_pos = positions[key]
                break
        for key in positions:
            if key != pick_obj and 'drawer' not in key:
                place_loc = positions[key]
                break

        if pick_obj is not None and pick_obj_pos is not None and place_loc is not None:
            print(f"[Task] Picking object '{pick_obj}' at {pick_obj_pos}")
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(pick_obj_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended during move to pick object!")
                    return

                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(pick_obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended during pick!")
                    return

                print(f"[Task] Placing object '{pick_obj}' at {place_loc}")
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(place_loc),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended during move to place location!")
                    return

                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(place_loc),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended during place!")
                    return

                print("[Task] Successfully picked and placed the object.")

            except Exception as e:
                print(f"[Task] Exception during pick and place: {e}")
        else:
            print("[Task] Could not find suitable pickable object and place location.")

        # === Drawer Manipulation Example (if available) ===
        # If a drawer and its side/anchor positions are available, try to rotate, move, and pull the drawer

        if drawer_name is not None:
            # Try to find side and anchor positions for the drawer
            side_pos = None
            anchor_pos = None
            for key in positions:
                if 'side' in key and drawer_name in key:
                    side_pos = positions[key]
                if 'anchor' in key and drawer_name in key:
                    anchor_pos = positions[key]
            if side_pos is not None and anchor_pos is not None:
                try:
                    print(f"[Drawer] Moving to side position {side_pos} of {drawer_name}")
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=np.array(side_pos),
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    if done:
                        print("[Drawer] Task ended during move to side position!")
                        return

                    print(f"[Drawer] Rotating gripper to 90 degrees for drawer manipulation")
                    obs, reward, done = rotate(
                        env,
                        task,
                        angle='ninety_deg'
                    )
                    if done:
                        print("[Drawer] Task ended during rotate!")
                        return

                    print(f"[Drawer] Moving to anchor position {anchor_pos} of {drawer_name}")
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=np.array(anchor_pos),
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    if done:
                        print("[Drawer] Task ended during move to anchor position!")
                        return

                    print(f"[Drawer] Pulling the drawer {drawer_name}")
                    obs, reward, done = pull(
                        env,
                        task,
                        drawer_name=drawer_name
                    )
                    if done:
                        print("[Drawer] Task ended during pull!")
                        return

                    print("[Drawer] Successfully manipulated the drawer.")

                except Exception as e:
                    print(f"[Drawer] Exception during drawer manipulation: {e}")
            else:
                print("[Drawer] Could not find side or anchor position for the drawer.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()