# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

# Constants for action parameters (replace magic numbers)
MAX_STEPS_ROTATE = 100
THRESHOLD_ROTATE = 0.05
TIMEOUT_ROTATE = 10.0
APPROACH_DISTANCE_PICK = 0.15
MAX_STEPS_PICK = 100
THRESHOLD_PICK = 0.01
TIMEOUT_PICK = 10.0

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Detect Missing Predicate/Knowledge ===
        # Feedback: Need to check if target object (e.g., 'tomato') is present and known
        #           Need to perform force calibration (weight-known), and obstacle avoidance

        # 1. Target object detection
        TARGET_OBJECT_NAME = 'tomato'
        target_object_pos = positions.get(TARGET_OBJECT_NAME, None)
        if target_object_pos is None:
            print(f"[Exploration] Target object '{TARGET_OBJECT_NAME}' not found in environment!")
            print("[Exploration] Aborting task.")
            return

        print(f"[Exploration] Target object '{TARGET_OBJECT_NAME}' detected at {target_object_pos}")

        # 2. Move to target object location to identify and sense it
        #    (simulate 'identified' and 'temperature-known' predicates)
        try:
            print(f"[Exploration] Moving to '{TARGET_OBJECT_NAME}' location for identification and temperature sensing.")
            obs, reward, done = move(
                env,
                task,
                target_pos=target_object_pos,
                approach_distance=APPROACH_DISTANCE_PICK,
                max_steps=MAX_STEPS_PICK,
                threshold=THRESHOLD_PICK,
                approach_axis='z',
                timeout=TIMEOUT_PICK
            )
            if done:
                print("[Exploration] Task ended during move!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")
            return

        # 3. Pick the object to perform force calibration (simulate 'weight-known' and 'durability-known')
        try:
            print(f"[Exploration] Picking up '{TARGET_OBJECT_NAME}' for force calibration (weight/durability sensing).")
            obs, reward, done = pick(
                env,
                task,
                target_pos=target_object_pos,
                approach_distance=APPROACH_DISTANCE_PICK,
                max_steps=MAX_STEPS_PICK,
                threshold=THRESHOLD_PICK,
                approach_axis='z',
                timeout=TIMEOUT_PICK
            )
            if done:
                print("[Exploration] Task ended during pick!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")
            return

        # 4. Obstacle avoidance check (simulate by checking for obstacles in path)
        #    (Assume get_object_positions() provides all objects; check for obstacles near path)
        OBSTACLE_DISTANCE_THRESHOLD = 0.10  # meters
        obstacles = []
        for obj_name, obj_pos in positions.items():
            if obj_name != TARGET_OBJECT_NAME:
                dist = np.linalg.norm(np.array(obj_pos) - np.array(target_object_pos))
                if dist < OBSTACLE_DISTANCE_THRESHOLD:
                    obstacles.append((obj_name, obj_pos))
        if obstacles:
            print(f"[Exploration] Obstacles detected near '{TARGET_OBJECT_NAME}': {obstacles}")
            print("[Exploration] Attempting to avoid obstacles.")
            # Here, you would call a predefined skill for obstacle avoidance if available.
            # Since only move/pick/place/rotate/pull are available, just log and proceed.
        else:
            print("[Exploration] No obstacles detected near target.")

        # 5. (Optional) Place the object at a disposal location if required
        DISPOSAL_LOCATION_NAME = 'disposal_bin'
        disposal_pos = positions.get(DISPOSAL_LOCATION_NAME, None)
        if disposal_pos is not None:
            try:
                print(f"[Task] Placing '{TARGET_OBJECT_NAME}' at disposal location '{DISPOSAL_LOCATION_NAME}'.")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=disposal_pos,
                    approach_distance=APPROACH_DISTANCE_PICK,
                    max_steps=MAX_STEPS_PICK,
                    threshold=THRESHOLD_PICK,
                    approach_axis='z',
                    timeout=TIMEOUT_PICK
                )
                if done:
                    print("[Task] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Task] Exception during place: {e}")
                return
        else:
            print(f"[Task] No disposal location '{DISPOSAL_LOCATION_NAME}' found. Skipping place step.")

        # 6. (Optional) Rotate or pull actions if required by the plan
        #    For demonstration, check if 'rotate' or 'pull' is needed for the object
        #    (Assume we want to rotate the gripper to a certain orientation before/after pick)
        #    Use a dummy target quaternion for demonstration
        TARGET_QUAT = np.array([0, 0, 0, 1])  # Identity quaternion
        try:
            print(f"[Task] Rotating gripper to target orientation for '{TARGET_OBJECT_NAME}'.")
            obs, reward, done = rotate(
                env,
                task,
                target_quat=TARGET_QUAT,
                max_steps=MAX_STEPS_ROTATE,
                threshold=THRESHOLD_ROTATE,
                timeout=TIMEOUT_ROTATE
            )
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Task] Exception during rotate: {e}")
            return

        # If the object is a drawer or requires pulling, call pull
        if 'drawer' in TARGET_OBJECT_NAME:
            try:
                print(f"[Task] Pulling '{TARGET_OBJECT_NAME}'.")
                obs, reward, done = pull(
                    env,
                    task,
                    target_name=TARGET_OBJECT_NAME
                )
                if done:
                    print("[Task] Task ended during pull!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pull: {e}")
                return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
