# run_skeleton_task.py (Completed with Exploration Phase and Feedback Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

import time
import traceback

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The goal is to explore the environment and try to discover which predicate is missing,
        # as per the feedback and exploration knowledge.

        # 1. Buffer for previous gripper pose (for consistent state updates)
        prev_gripper_pose = None

        # 2. Safety thresholds (example values, adjust as needed)
        WORKSPACE_BOUNDS = {
            'x': (-1.0, 1.0),
            'y': (-1.0, 1.0),
            'z': (0.0, 1.5)
        }

        # 3. Object list review
        object_names = list(positions.keys())
        print("[Exploration] Objects in environment:", object_names)

        # 4. Error classification helper
        def classify_and_report_error(e):
            print("[Error] Exception occurred:", str(e))
            traceback.print_exc()
            if isinstance(e, KeyError):
                print("[Error] Likely cause: Object or key not found in positions or environment.")
            elif isinstance(e, TypeError):
                print("[Error] Likely cause: Type mismatch in skill function arguments.")
            else:
                print("[Error] Unclassified error.")

        # 5. Safety check for gripper pose
        def is_within_workspace(pos):
            return (WORKSPACE_BOUNDS['x'][0] <= pos[0] <= WORKSPACE_BOUNDS['x'][1] and
                    WORKSPACE_BOUNDS['y'][0] <= pos[1] <= WORKSPACE_BOUNDS['y'][1] and
                    WORKSPACE_BOUNDS['z'][0] <= pos[2] <= WORKSPACE_BOUNDS['z'][1])

        # 6. Exploration: Try to pick each object and observe for missing predicate
        for obj_name in object_names:
            try:
                print(f"[Exploration] Attempting to pick object: {obj_name}")
                obj_pos = positions[obj_name]
                # Safety check
                if not is_within_workspace(obj_pos):
                    print(f"[Safety] Object {obj_name} at {obj_pos} is out of workspace bounds. Skipping.")
                    continue

                # Try to pick the object
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )

                # Buffer previous gripper pose for consistent state update
                prev_gripper_pose = obs.gripper_pose if hasattr(obs, 'gripper_pose') else None

                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}.")
                    break

                # After pick, try to place the object back (if possible)
                print(f"[Exploration] Attempting to place object: {obj_name}")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )

                if done:
                    print(f"[Exploration] Task ended after placing {obj_name}.")
                    break

            except Exception as e:
                classify_and_report_error(e)
                continue

        # 7. Exploration: Try to rotate and pull (if available in skills and environment)
        for obj_name in object_names:
            try:
                print(f"[Exploration] Attempting to rotate gripper near object: {obj_name}")
                obj_pos = positions[obj_name]
                # Safety check
                if not is_within_workspace(obj_pos):
                    print(f"[Safety] Object {obj_name} at {obj_pos} is out of workspace bounds. Skipping.")
                    continue

                # Example: Try to rotate to a fixed quaternion (identity)
                target_quat = np.array([0, 0, 0, 1])
                obs, reward, done = rotate(
                    env,
                    task,
                    target_quat=target_quat,
                    max_steps=50,
                    threshold=0.05,
                    timeout=5.0
                )

                prev_gripper_pose = obs.gripper_pose if hasattr(obs, 'gripper_pose') else None

                if done:
                    print(f"[Exploration] Task ended after rotating near {obj_name}.")
                    break

                # Try to pull (if the object is a drawer or has a handle)
                if 'drawer' in obj_name or 'handle' in obj_name:
                    print(f"[Exploration] Attempting to pull object: {obj_name}")
                    obs, reward, done = pull(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after pulling {obj_name}.")
                        break

            except Exception as e:
                classify_and_report_error(e)
                continue

        # 8. Exploration: Try to move to each object location (if move skill is available)
        if 'move' in ['pick', 'place', 'move', 'rotate', 'pull']:
            for obj_name in object_names:
                try:
                    print(f"[Exploration] Attempting to move to object: {obj_name}")
                    obj_pos = positions[obj_name]
                    if not is_within_workspace(obj_pos):
                        print(f"[Safety] Object {obj_name} at {obj_pos} is out of workspace bounds. Skipping.")
                        continue

                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    prev_gripper_pose = obs.gripper_pose if hasattr(obs, 'gripper_pose') else None

                    if done:
                        print(f"[Exploration] Task ended after moving to {obj_name}.")
                        break

                except Exception as e:
                    classify_and_report_error(e)
                    continue

        print("[Exploration] Completed exploration phase. If any missing predicate caused an error, it should be visible above.")

        # === End of Exploration ===

        # (Optional) Here you could implement further logic to analyze the results of the exploration
        # and deduce which predicate is missing, based on which actions failed or which state transitions
        # did not occur as expected.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
