# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def euler_from_quat(quaternion):
    # Assumes quaternion in [w, x, y, z] order
    w, x, y, z = quaternion
    ysqr = y * y

    t0 = +2.0 * (w * x + y * z)
    t1 = +1.0 - 2.0 * (x * x + ysqr)
    roll = np.arctan2(t0, t1)

    t2 = +2.0 * (w * y - z * x)
    t2 = np.clip(t2, -1.0, 1.0)
    pitch = np.arcsin(t2)

    t3 = +2.0 * (w * z + x * y)
    t4 = +1.0 - 2.0 * (ysqr + z * z)
    yaw = np.arctan2(t3, t4)

    return np.array([roll, pitch, yaw])

def normalize_quaternion(quaternion):
    # Assumes quaternion in [w, x, y, z] order
    norm = np.linalg.norm(quaternion)
    if norm == 0:
        return np.array([1.0, 0.0, 0.0, 0.0])
    return np.array(quaternion) / norm

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The goal is to explore the environment to determine which predicate is missing.
        # We use the available skills: move, pick, place, rotate, pull.
        # The exploration domain suggests that moving to a location can help identify objects,
        # and picking can help determine weight/durability, and pulling can help determine lock state.

        # For demonstration, we will:
        # 1. Move to each location and try to identify objects.
        # 2. Try to pick each object to see if weight/durability is known.
        # 3. Try to pull drawers/objects to see if lock state is known.

        # The actual object and location names depend on the environment.
        # We'll use the keys from positions as object/location names.

        # Example structure of positions:
        # positions = {
        #   'object_1': {'type': 'object', 'position': (x, y, z), 'location': 'loc1'},
        #   'drawer_1': {'type': 'drawer', 'position': (x, y, z), 'location': 'loc2'},
        #   ...
        # }

        # For this exploration, we need to:
        # - Move to each location (if possible)
        # - For each object at the location, try pick/pull as appropriate

        # We'll keep track of which predicates we "discover" by attempting actions

        discovered_predicates = set()
        failed_actions = []

        # 1. Move to each location to identify objects
        location_names = set()
        for obj_name, info in positions.items():
            if 'location' in info:
                location_names.add(info['location'])
        location_names = list(location_names)

        # Assume the robot starts at the first location
        if len(location_names) == 0:
            print("[Exploration] No locations found in object_positions.")
        else:
            current_location = location_names[0]
            robot_name = 'robot'  # Placeholder, adjust if your env uses a different robot name

            for loc in location_names:
                try:
                    print(f"[Exploration] Moving to location: {loc}")
                    obs, reward, done = move(env, task, robot_name, current_location, loc)
                    current_location = loc
                    # After moving, we assume all objects at this location become 'identified'
                    for obj_name, info in positions.items():
                        if info.get('location', None) == loc:
                            discovered_predicates.add(f"identified({obj_name})")
                except Exception as e:
                    print(f"[Exploration] Failed to move to {loc}: {e}")
                    failed_actions.append(('move', loc, str(e)))

        # 2. Try to pick each object at its location
        for obj_name, info in positions.items():
            obj_type = info.get('type', 'object')
            obj_loc = info.get('location', None)
            if obj_loc is None:
                continue
            try:
                print(f"[Exploration] Attempting to pick object: {obj_name} at {obj_loc}")
                obs, reward, done = pick(env, task, robot_name, obj_name, obj_loc)
                # If pick succeeds, we assume weight/durability may be discovered
                discovered_predicates.add(f"weight-known({obj_name})")
                discovered_predicates.add(f"durability-known({obj_name})")
                # Optionally, place the object back
                print(f"[Exploration] Placing object: {obj_name} at {obj_loc}")
                obs, reward, done = place(env, task, robot_name, obj_name, obj_loc)
            except Exception as e:
                print(f"[Exploration] Failed to pick/place {obj_name}: {e}")
                failed_actions.append(('pick/place', obj_name, str(e)))

        # 3. Try to pull drawers/objects to see if lock state is known
        for obj_name, info in positions.items():
            obj_type = info.get('type', '')
            obj_loc = info.get('location', None)
            if obj_type == 'drawer' and obj_loc is not None:
                try:
                    print(f"[Exploration] Attempting to pull drawer: {obj_name} at {obj_loc}")
                    obs, reward, done = pull(env, task, robot_name, obj_name, obj_loc)
                    discovered_predicates.add(f"lock-known({obj_name})")
                except Exception as e:
                    print(f"[Exploration] Failed to pull {obj_name}: {e}")
                    failed_actions.append(('pull', obj_name, str(e)))

        # 4. Try to rotate gripper to see if any orientation predicates are discovered
        # (Assume we have a gripper name and target quaternion)
        gripper_name = None
        for obj_name, info in positions.items():
            if info.get('type', '') == 'gripper':
                gripper_name = obj_name
                break
        if gripper_name is not None:
            # Try rotating to a sample quaternion (identity)
            target_quat = [1.0, 0.0, 0.0, 0.0]
            try:
                print(f"[Exploration] Rotating gripper: {gripper_name}")
                obs, reward, done = rotate(env, task, target_quat)
                discovered_predicates.add(f"rotated({gripper_name})")
            except Exception as e:
                print(f"[Exploration] Failed to rotate {gripper_name}: {e}")
                failed_actions.append(('rotate', gripper_name, str(e)))

        # === Exploration Summary ===
        print("\n[Exploration] Discovered predicates:")
        for pred in discovered_predicates:
            print("  ", pred)
        if failed_actions:
            print("\n[Exploration] Failed actions:")
            for act in failed_actions:
                print("  ", act)

        # === End of Exploration Phase ===
        # At this point, you can analyze discovered_predicates to infer which predicate(s) are missing
        # or not being set as expected, based on the domain and the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
