# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Info] Object positions:", positions)

        # === Exploration Phase: Try to Discover Missing Predicate ===
        # The feedback suggests a missing predicate is causing planning to fail.
        # We will attempt to use available skills in a systematic way to explore the environment
        # and print out observations after each action to help identify what predicate might be missing.

        # For demonstration, we will:
        # 1. Try to move to each location (if move is available)
        # 2. Try to pick and place each object (if possible)
        # 3. Try to rotate and pull (if possible)
        # 4. Print out the observation after each action

        # Get all objects and locations from positions
        object_names = []
        location_names = []
        for name, pos in positions.items():
            if "drawer" in name or "gripper" in name or "angle" in name or "position" in name:
                continue  # skip non-movable objects for pick/place
            object_names.append(name)
        # For this example, we assume locations are provided as keys with 'location' in their name
        for name in positions.keys():
            if "location" in name or "room" in name or "table" in name:
                location_names.append(name)
        if not location_names:
            # Fallback: use all unique locations from object positions
            for obj in object_names:
                loc = positions[obj]
                if isinstance(loc, str) and loc not in location_names:
                    location_names.append(loc)

        # 1. Try to move to each location (if 'move' is available)
        if 'move' in ['pick', 'place', 'move', 'rotate', 'pull']:
            print("[Exploration] Attempting to move to each location (if possible)...")
            for loc in location_names:
                try:
                    print(f"[Exploration] Moving to location: {loc}")
                    obs, reward, done = move(env, task, target_location=loc)
                    print(f"[Exploration] After move to {loc}, obs: {obs}")
                    if done:
                        print("[Exploration] Task ended during move!")
                        return
                except Exception as e:
                    print(f"[Exploration] Move to {loc} failed: {e}")

        # 2. Try to pick and place each object at each location
        if 'pick' in ['pick', 'place', 'move', 'rotate', 'pull']:
            print("[Exploration] Attempting to pick and place each object...")
            for obj in object_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying to pick {obj} at {loc}")
                        obs, reward, done = pick(env, task, target_obj=obj, target_location=loc)
                        print(f"[Exploration] After pick {obj} at {loc}, obs: {obs}")
                        if done:
                            print("[Exploration] Task ended during pick!")
                            return
                        # Try to place at another location if possible
                        for place_loc in location_names:
                            if place_loc == loc:
                                continue
                            try:
                                print(f"[Exploration] Trying to place {obj} at {place_loc}")
                                obs, reward, done = place(env, task, target_obj=obj, target_location=place_loc)
                                print(f"[Exploration] After place {obj} at {place_loc}, obs: {obs}")
                                if done:
                                    print("[Exploration] Task ended during place!")
                                    return
                            except Exception as e:
                                print(f"[Exploration] Place {obj} at {place_loc} failed: {e}")
                    except Exception as e:
                        print(f"[Exploration] Pick {obj} at {loc} failed: {e}")

        # 3. Try to rotate and pull (if possible)
        if 'rotate' in ['pick', 'place', 'move', 'rotate', 'pull']:
            print("[Exploration] Attempting to rotate grippers (if any)...")
            for name in positions.keys():
                if "gripper" in name:
                    try:
                        # For demonstration, try rotating to a fixed quaternion (e.g., [0,0,0,1])
                        print(f"[Exploration] Rotating {name} to [0,0,0,1]")
                        obs, reward, done = rotate(env, task, target_quat=[0,0,0,1])
                        print(f"[Exploration] After rotate {name}, obs: {obs}")
                        if done:
                            print("[Exploration] Task ended during rotate!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Rotate {name} failed: {e}")

        if 'pull' in ['pick', 'place', 'move', 'rotate', 'pull']:
            print("[Exploration] Attempting to pull objects (if any)...")
            for obj in object_names:
                try:
                    print(f"[Exploration] Trying to pull {obj}")
                    obs, reward, done = pull(env, task, target_obj=obj)
                    print(f"[Exploration] After pull {obj}, obs: {obs}")
                    if done:
                        print("[Exploration] Task ended during pull!")
                        return
                except Exception as e:
                    print(f"[Exploration] Pull {obj} failed: {e}")

        print("[Exploration] Exploration phase complete. Review the above observations to identify missing predicates.")

        # === End of Exploration Phase ===

        # TODO: After identifying the missing predicate, update the domain/problem and re-run the plan.
        # For now, the code only explores and prints observations to help the user identify the missing predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()